# PDF Placeholder System Design Document

## Overview

This design document outlines the comprehensive PDF template management system that allows users to upload PDF templates, interactively place placeholders at specific positions, configure placeholder properties, and generate certificates by filling placeholders with dynamic data. The system builds upon the existing certificate generation infrastructure while adding enhanced template editing capabilities.

## Architecture

The solution implements a layered architecture that extends the current system:

1. **Template Upload Layer**: Enhanced PDF upload with validation and storage
2. **Interactive Editor Layer**: Browser-based PDF viewer with placeholder positioning
3. **Placeholder Management Layer**: Configuration and storage of placeholder properties
4. **Certificate Generation Layer**: Integration with existing PDF processing services
5. **Print Management Layer**: Optimized output for printing

## Components and Interfaces

### 1. Enhanced Template Manager

**Location**: `src/Service/TemplateManager.php`

```php
interface TemplateManagerInterface
{
    public function uploadTemplate(array $fileData, array $templateConfig): TemplateUploadResult;
    public function updatePlaceholders(int $templateId, array $placeholders): bool;
    public function getTemplateWithPlaceholders(int $templateId): TemplateData;
    public function validateTemplateStructure(string $templatePath): ValidationResult;
}
```

**Responsibilities**:
- Handle PDF template uploads with validation
- Store template metadata and placeholder configurations
- Provide template data for editing and generation
- Integrate with existing validation services

### 2. Interactive PDF Editor

**Location**: `public/js/pdf-editor.js`

**Components**:
- PDF.js integration for rendering
- Interactive placeholder positioning
- Real-time coordinate capture
- Drag-and-drop placeholder management
- Zoom and pan functionality

**Responsibilities**:
- Render PDF templates in browser
- Capture precise click coordinates
- Provide visual feedback for placeholder placement
- Handle placeholder property editing
- Save placeholder configurations

### 3. Placeholder Service

**Location**: `src/Service/PlaceholderService.php`

```php
interface PlaceholderServiceInterface
{
    public function createPlaceholder(array $placeholderData): Placeholder;
    public function updatePlaceholder(int $placeholderId, array $updates): bool;
    public function deletePlaceholder(int $placeholderId): bool;
    public function getPlaceholdersForTemplate(int $templateId): array;
    public function validatePlaceholderConfiguration(array $config): ValidationResult;
}
```

**Responsibilities**:
- Manage placeholder CRUD operations
- Validate placeholder configurations
- Handle coordinate transformations
- Store placeholder properties and styling

### 4. Enhanced Certificate Generator

**Location**: `src/Service/CertificateGenerator.php`

```php
interface CertificateGeneratorInterface
{
    public function generateCertificate(int $templateId, array $fieldData): CertificateResult;
    public function generateBatchCertificates(int $templateId, array $batchData): BatchResult;
    public function previewCertificate(int $templateId, array $fieldData): PreviewResult;
    public function preparePrintOutput(string $certificateContent): PrintResult;
}
```

**Responsibilities**:
- Generate certificates using template placeholders
- Handle batch certificate generation
- Provide print-optimized output
- Integrate with existing PDF processing services

### 5. Print Manager

**Location**: `src/Service/PrintManager.php`

```php
interface PrintManagerInterface
{
    public function optimizeForPrint(string $pdfContent): string;
    public function validatePrintSettings(array $settings): ValidationResult;
    public function generatePrintPreview(string $pdfContent, array $settings): PreviewResult;
}
```

**Responsibilities**:
- Optimize PDF output for printing
- Handle print settings and paper sizes
- Provide print preview functionality
- Ensure print quality and resolution

## Data Models

### Enhanced Template Model

```php
class Template
{
    public int $id;
    public string $name;
    public string $useCase;
    public string $filePath;
    public array $fieldDefinitions;
    public array $placeholders;
    public array $printSettings;
    public string $createdBy;
    public DateTime $createdAt;
    public DateTime $updatedAt;
    public string $status;
    public ?array $validationData;
}
```

### Placeholder Model

```php
class Placeholder
{
    public int $id;
    public int $templateId;
    public string $fieldName;
    public string $type; // 'text', 'date', 'image', 'signature'
    public float $x;
    public float $y;
    public float $width;
    public float $height;
    public int $page;
    public array $styling; // font, size, color, alignment
    public array $validation; // required, format, maxLength
    public bool $isRequired;
    public ?string $defaultValue;
    public DateTime $createdAt;
    public DateTime $updatedAt;
}
```

### Certificate Generation Result

```php
class CertificateResult
{
    public bool $success;
    public ?string $pdfContent;
    public ?string $filePath;
    public array $metadata;
    public ?string $errorMessage;
    public array $placeholderData;
    public string $generationMethod; // 'fpdi', 'fallback', 'tcpdf'
}
```

## Database Schema Extensions

### New Placeholders Table

```sql
CREATE TABLE certificate_placeholders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    template_id INT NOT NULL,
    field_name VARCHAR(100) NOT NULL,
    placeholder_type ENUM('text', 'date', 'image', 'signature') DEFAULT 'text',
    x_coordinate DECIMAL(10,4) NOT NULL,
    y_coordinate DECIMAL(10,4) NOT NULL,
    width DECIMAL(10,4) DEFAULT 0,
    height DECIMAL(10,4) DEFAULT 0,
    page_number INT DEFAULT 1,
    font_family VARCHAR(50) DEFAULT 'helvetica',
    font_size INT DEFAULT 12,
    font_color VARCHAR(7) DEFAULT '#000000',
    text_alignment ENUM('left', 'center', 'right') DEFAULT 'left',
    is_required BOOLEAN DEFAULT TRUE,
    default_value TEXT NULL,
    validation_rules JSON NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (template_id) REFERENCES certificate_templates(id) ON DELETE CASCADE,
    INDEX idx_template_placeholders (template_id),
    INDEX idx_field_name (field_name)
);
```

### Enhanced Templates Table

```sql
ALTER TABLE certificate_templates 
ADD COLUMN print_settings JSON NULL,
ADD COLUMN template_version INT DEFAULT 1,
ADD COLUMN last_edited_by VARCHAR(255) NULL,
ADD COLUMN last_edited_at TIMESTAMP NULL;
```

## User Interface Design

### Template Upload Interface

**Enhanced Upload Form**:
- PDF file selection with drag-and-drop
- Template name and description
- Use case type selection
- Field definitions (comma-separated or individual inputs)
- Automatic validation feedback
- Upload progress indicator

### Interactive PDF Editor

**Editor Components**:
- Full-screen PDF viewer with zoom controls
- Placeholder toolbar with field types
- Properties panel for selected placeholders
- Coordinate display and manual adjustment
- Save/cancel actions
- Preview mode toggle

**Placeholder Interaction**:
- Click to place new placeholders
- Drag to reposition existing placeholders
- Resize handles for dimension adjustment
- Context menu for placeholder actions
- Visual indicators for placeholder types

### Certificate Generation Interface

**Generation Form**:
- Template selection dropdown
- Dynamic form fields based on placeholders
- Field validation and error display
- Preview button for certificate preview
- Generate and download actions
- Batch generation options

## Integration with Existing System

### PDF Processing Integration

The new system integrates seamlessly with the existing PDF processing infrastructure:

1. **Template Validation**: Uses existing `TemplateValidator` service
2. **PDF Generation**: Extends existing `PdfProcessor` with placeholder support
3. **Error Handling**: Leverages existing `ErrorHandler` for consistent error management
4. **Fallback Support**: Maintains compatibility with fallback generation methods

### Database Integration

- Extends existing `certificate_templates` table
- Maintains foreign key relationships
- Preserves existing certificate generation workflow
- Adds new placeholder management capabilities

### Service Factory Integration

```php
class ServiceFactory
{
    public static function getTemplateManager(): TemplateManagerInterface
    {
        return new TemplateManager(
            self::getPlaceholderService(),
            self::getTemplateValidator(),
            self::getErrorHandler()
        );
    }
    
    public static function getPlaceholderService(): PlaceholderServiceInterface
    {
        return new PlaceholderService(getDbConnection());
    }
    
    public static function getCertificateGenerator(): CertificateGeneratorInterface
    {
        return new CertificateGenerator(
            self::getPdfProcessor(),
            self::getPlaceholderService(),
            self::getPrintManager()
        );
    }
}
```

## Security Considerations

### File Upload Security
- Validate PDF file types and signatures
- Implement file size limits
- Scan for malicious content
- Store uploads outside web root
- Generate unique file names

### Coordinate Validation
- Validate coordinate ranges within PDF bounds
- Sanitize coordinate input data
- Prevent coordinate injection attacks
- Validate page number ranges

### Access Control
- Restrict template editing to authorized users
- Implement role-based permissions
- Log template modification activities
- Validate user sessions for all operations

## Performance Considerations

### PDF Rendering Optimization
- Implement client-side PDF caching
- Use progressive loading for large PDFs
- Optimize canvas rendering performance
- Implement viewport-based rendering

### Database Optimization
- Index placeholder queries by template_id
- Implement placeholder caching
- Optimize coordinate storage format
- Use prepared statements for all queries

### Memory Management
- Stream large PDF files
- Implement garbage collection for PDF objects
- Limit concurrent PDF processing
- Monitor memory usage during generation

## Print Optimization

### Print Quality Settings
- Support high-resolution output (300+ DPI)
- Maintain vector graphics quality
- Optimize font rendering for print
- Handle color profile conversions

### Paper Size Support
- Support standard paper sizes (A4, Letter, Legal)
- Handle custom paper dimensions
- Maintain aspect ratio during scaling
- Provide print preview with margins

### Batch Printing
- Generate print-ready batch files
- Optimize memory usage for large batches
- Provide progress feedback
- Handle print queue management

## Testing Strategy

### Unit Testing
- Test placeholder CRUD operations
- Validate coordinate transformations
- Test PDF processing with placeholders
- Mock external dependencies

### Integration Testing
- Test complete template upload workflow
- Validate certificate generation pipeline
- Test print output quality
- Verify database transactions

### User Interface Testing
- Test PDF editor interactions
- Validate placeholder positioning accuracy
- Test responsive design elements
- Verify cross-browser compatibility

### Performance Testing
- Load test with large PDF files
- Stress test placeholder positioning
- Test concurrent user scenarios
- Validate memory usage patterns

## Implementation Phases

### Phase 1: Core Infrastructure
- Implement placeholder data models
- Create database schema extensions
- Build basic placeholder service
- Extend template manager

### Phase 2: Interactive Editor
- Integrate PDF.js for rendering
- Implement placeholder positioning
- Build properties configuration panel
- Add coordinate capture functionality

### Phase 3: Certificate Generation
- Extend PDF processor for placeholders
- Implement certificate generation with placeholders
- Add preview functionality
- Integrate with existing generation workflow

### Phase 4: Print Optimization
- Implement print manager service
- Add print quality optimization
- Build print preview functionality
- Support batch printing operations

### Phase 5: User Interface Polish
- Enhance template upload interface
- Improve editor user experience
- Add advanced placeholder features
- Implement comprehensive error handling

## Monitoring and Analytics

### Usage Tracking
- Track template creation and usage
- Monitor placeholder positioning accuracy
- Measure certificate generation success rates
- Analyze print quality feedback

### Performance Monitoring
- Monitor PDF processing times
- Track memory usage patterns
- Measure editor responsiveness
- Alert on system performance issues

### Error Tracking
- Log placeholder configuration errors
- Track PDF processing failures
- Monitor print quality issues
- Provide detailed error reporting