-- ========================================
-- CertifyEd Database Schema
-- Complete database structure for certificate management system
-- ========================================

/*!40101 SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT */;
/*!40101 SET NAMES utf8 */;
/*!50503 SET NAMES utf8mb4 */;
/*!40103 SET @OLD_TIME_ZONE=@@TIME_ZONE */;
/*!40103 SET TIME_ZONE='+00:00' */;
/*!40014 SET @OLD_FOREIGN_KEY_CHECKS=@@FOREIGN_KEY_CHECKS, FOREIGN_KEY_CHECKS=0 */;
/*!40101 SET @OLD_SQL_MODE=@@SQL_MODE, SQL_MODE='NO_AUTO_VALUE_ON_ZERO' */;
/*!40111 SET @OLD_SQL_NOTES=@@SQL_NOTES, SQL_NOTES=0 */;

-- Create database
CREATE DATABASE IF NOT EXISTS `certifyed` /*!40100 DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_0900_ai_ci */ /*!80016 DEFAULT ENCRYPTION='N' */;
USE `certifyed`;

-- ========================================
-- Table: users
-- Manages system users (admins, backend users)
-- ========================================
CREATE TABLE IF NOT EXISTS `users` (
  `id` int NOT NULL AUTO_INCREMENT,
  `name` varchar(255) NOT NULL COMMENT 'Full name of the user',
  `email` varchar(255) NOT NULL COMMENT 'Unique email address for login',
  `role` enum('super_admin','admin','backend_user') NOT NULL COMMENT 'User role and permissions level',
  `password_hash` varchar(255) NOT NULL COMMENT 'Hashed password for authentication',
  `status` varchar(20) NOT NULL DEFAULT 'active' COMMENT 'Account status (active, inactive, suspended)',
  PRIMARY KEY (`id`),
  UNIQUE KEY `email` (`email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci COMMENT='System users and authentication';

-- ========================================
-- Table: certificate_templates
-- Stores PDF templates and their field configurations
-- ========================================
CREATE TABLE IF NOT EXISTS `certificate_templates` (
  `id` int NOT NULL AUTO_INCREMENT,
  `name` varchar(255) NOT NULL COMMENT 'Template display name',
  `use_case_type` enum('protected','general_teacher','student') NOT NULL COMMENT 'Template category/type',
  `file_path` varchar(255) NOT NULL COMMENT 'Path to PDF template file',
  `field_definitions` json NOT NULL COMMENT 'Array of field names required for this template',
  `field_coordinates` text COMMENT 'JSON data for field positioning on PDF',
  `validation_data` json DEFAULT NULL COMMENT 'Template validation and processing metadata',
  `created_by` varchar(255) NOT NULL DEFAULT '' COMMENT 'User who created this template',
  `status` varchar(20) NOT NULL DEFAULT 'active' COMMENT 'Template status (active, inactive, deleted)',
  PRIMARY KEY (`id`),
  KEY `idx_use_case_status` (`use_case_type`, `status`),
  KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci COMMENT='PDF certificate templates and configurations';

-- ========================================
-- Table: certificates
-- Generated certificates and download tracking
-- ========================================
CREATE TABLE IF NOT EXISTS `certificates` (
  `id` int NOT NULL AUTO_INCREMENT,
  `template_id` int NOT NULL COMMENT 'Reference to certificate template used',
  `email` varchar(255) NOT NULL COMMENT 'Email address of certificate recipient',
  `file_path` varchar(255) NOT NULL COMMENT 'Path to generated certificate PDF',
  `license_code` varchar(64) DEFAULT NULL COMMENT 'Unique license code for secure downloads',
  `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'When certificate was generated',
  `download_used` tinyint(1) NOT NULL DEFAULT '0' COMMENT 'Whether download link has been used',
  `download_created_at` datetime DEFAULT NULL COMMENT 'When download link was created',
  PRIMARY KEY (`id`),
  KEY `template_id` (`template_id`),
  KEY `idx_license_code` (`license_code`),
  KEY `idx_email` (`email`),
  KEY `idx_created_at` (`created_at`),
  CONSTRAINT `certificates_ibfk_1` FOREIGN KEY (`template_id`) REFERENCES `certificate_templates` (`id`) ON DELETE RESTRICT ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci COMMENT='Generated certificates and download tracking';

-- ========================================
-- Table: license_codes
-- Pre-generated license codes for secure certificate access
-- ========================================
CREATE TABLE IF NOT EXISTS `license_codes` (
  `id` int NOT NULL AUTO_INCREMENT,
  `code` varchar(64) NOT NULL COMMENT 'Unique license code',
  `email` varchar(255) DEFAULT NULL COMMENT 'Associated email address',
  `template_id` int NOT NULL COMMENT 'Template this license is valid for',
  `status` enum('used','unused','expired') NOT NULL DEFAULT 'unused' COMMENT 'License code status',
  `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'When license was created',
  `used_at` datetime DEFAULT NULL COMMENT 'When license was used',
  `download_created_at` datetime DEFAULT NULL COMMENT 'When download was initiated',
  `created_by` varchar(255) DEFAULT NULL COMMENT 'Admin who created this license',
  `token` varchar(64) DEFAULT NULL COMMENT 'Additional security token',
  PRIMARY KEY (`id`),
  UNIQUE KEY `code` (`code`),
  UNIQUE KEY `token` (`token`),
  KEY `template_id` (`template_id`),
  KEY `idx_status` (`status`),
  KEY `idx_email` (`email`),
  CONSTRAINT `license_codes_ibfk_1` FOREIGN KEY (`template_id`) REFERENCES `certificate_templates` (`id`) ON DELETE RESTRICT ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci COMMENT='Pre-generated license codes for secure access';

-- ========================================
-- Table: certificate_access_log
-- Audit trail for certificate downloads and access
-- ========================================
CREATE TABLE IF NOT EXISTS `certificate_access_log` (
  `id` int NOT NULL AUTO_INCREMENT,
  `certificate_id` int NOT NULL COMMENT 'Reference to accessed certificate',
  `accessed_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'When certificate was accessed',
  `ip_address` varchar(45) DEFAULT NULL COMMENT 'IP address of accessor (IPv4/IPv6)',
  `user_email` varchar(255) DEFAULT NULL COMMENT 'Email of user who accessed certificate',
  PRIMARY KEY (`id`),
  KEY `certificate_id` (`certificate_id`),
  KEY `idx_accessed_at` (`accessed_at`),
  KEY `idx_user_email` (`user_email`),
  CONSTRAINT `certificate_access_log_ibfk_1` FOREIGN KEY (`certificate_id`) REFERENCES `certificates` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci COMMENT='Audit trail for certificate access';

-- ========================================
-- Table: logs
-- General system activity logging
-- ========================================
CREATE TABLE IF NOT EXISTS `logs` (
  `id` int NOT NULL AUTO_INCREMENT,
  `user_id` int DEFAULT NULL COMMENT 'User who performed the action',
  `action` varchar(255) NOT NULL COMMENT 'Description of action performed',
  `target_id` int DEFAULT NULL COMMENT 'ID of affected resource',
  `timestamp` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'When action occurred',
  PRIMARY KEY (`id`),
  KEY `user_id` (`user_id`),
  KEY `idx_timestamp` (`timestamp`),
  KEY `idx_action` (`action`),
  CONSTRAINT `logs_ibfk_1` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_ai_ci COMMENT='System activity and audit logging';

-- ========================================
-- Indexes for Performance Optimization
-- ========================================

-- Additional indexes for common queries
ALTER TABLE `certificates` ADD INDEX `idx_template_email` (`template_id`, `email`);
ALTER TABLE `certificates` ADD INDEX `idx_download_status` (`download_used`, `download_created_at`);
ALTER TABLE `license_codes` ADD INDEX `idx_template_status` (`template_id`, `status`);

-- ========================================
-- Sample Data (Optional - for development)
-- ========================================

-- Insert default admin user (password: admin123)
-- Note: Change password in production!
INSERT IGNORE INTO `users` (`id`, `name`, `email`, `role`, `password_hash`, `status`) VALUES
(1, 'System Administrator', 'admin@certifyed.com', 'super_admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'active');

-- ========================================
-- Database Schema Information
-- ========================================

/*
Database: certifyed
Version: 2.0
Created: 2025-11-01
Last Updated: 2025-11-01

Tables:
1. users - System user management and authentication
2. certificate_templates - PDF templates and field configurations  
3. certificates - Generated certificates and download tracking
4. license_codes - Pre-generated secure access codes
5. certificate_access_log - Audit trail for certificate access
6. logs - General system activity logging

Key Features:
- Foreign key constraints for data integrity
- Comprehensive indexing for performance
- JSON fields for flexible metadata storage
- Audit trails for security and compliance
- Support for multiple certificate types
- Secure download mechanisms
- User role-based access control

Use Cases Supported:
- Protected certificates (admin-controlled)
- General teacher certificates (public access)
- Student certificates (bulk generation)
- License-based secure downloads
- Template management and validation
- User authentication and authorization
*/

/*!40103 SET TIME_ZONE=IFNULL(@OLD_TIME_ZONE, 'system') */;
/*!40101 SET SQL_MODE=IFNULL(@OLD_SQL_MODE, '') */;
/*!40014 SET FOREIGN_KEY_CHECKS=IFNULL(@OLD_FOREIGN_KEY_CHECKS, 1) */;
/*!40101 SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT */;
/*!40111 SET SQL_NOTES=IFNULL(@OLD_SQL_NOTES, 1) */;
