<?php
session_start();
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../config/config.php';

use App\Service\ServiceFactory;

// Check if user is admin
if (empty($_SESSION['admin_logged_in'])) {
    echo "<script>window.top.location.href = 'login.php';</script>";
    exit;
}

$pdo = getDbConnection();
$results = [];
$error = '';
$success = '';

// Handle batch validation request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['batch_validate'])) {
    try {
        // Get all active templates
        $stmt = $pdo->query("SELECT * FROM certificate_templates WHERE status = 'active' ORDER BY id");
        $templates = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $templateValidator = ServiceFactory::getTemplateValidator();
        $templatePaths = [];
        
        // Prepare template paths for batch validation
        foreach ($templates as $template) {
            $templatePath = __DIR__ . '/../templates/' . $template['file_path'];
            if (file_exists($templatePath)) {
                $templatePaths[] = $templatePath;
            }
        }
        
        // Perform batch validation
        $batchResults = $templateValidator->batchValidate($templatePaths);
        
        // Update database with results
        $updateCount = 0;
        foreach ($templates as $template) {
            $templatePath = __DIR__ . '/../templates/' . $template['file_path'];
            $templateName = basename($templatePath);
            
            if (isset($batchResults[$templateName])) {
                $validationResult = $batchResults[$templateName]['validation'];
                $quickTest = $batchResults[$templateName]['quickTest'];
                
                $validationData = json_encode([
                    'is_valid' => $validationResult->isValid,
                    'fpdi_compatible' => $validationResult->fpdiCompatible,
                    'issues' => $validationResult->issues,
                    'recommendations' => $validationResult->recommendations,
                    'validated_at' => date('Y-m-d H:i:s'),
                    'test_duration_ms' => $quickTest['testDuration'],
                    'file_size_bytes' => $quickTest['templateSize']
                ]);
                
                $stmt = $pdo->prepare("UPDATE certificate_templates SET validation_data = ? WHERE id = ?");
                $stmt->execute([$validationData, $template['id']]);
                $updateCount++;
                
                $results[] = [
                    'template' => $template,
                    'validation' => $validationResult,
                    'quickTest' => $quickTest
                ];
            }
        }
        
        $success = "Batch validation completed. Updated {$updateCount} templates.";
        
    } catch (Exception $e) {
        error_log("Batch validation error: " . $e->getMessage());
        $error = "Batch validation failed: " . $e->getMessage();
    }
}

// Get current validation status
$stmt = $pdo->query("SELECT id, name, file_path, validation_data FROM certificate_templates WHERE status = 'active' ORDER BY id");
$templates = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html>
<head>
    <title>Batch Template Validation - CertifyEd</title>
    <link rel="stylesheet" href="style.css">
    <style>
        .validation-summary {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
            margin: 20px 0;
        }
        .validation-card {
            background: #fff;
            border-radius: 8px;
            padding: 16px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            border-left: 4px solid #ddd;
        }
        .validation-card.compatible { border-left-color: #28a745; }
        .validation-card.fallback { border-left-color: #ffc107; }
        .validation-card.invalid { border-left-color: #dc3545; }
        .validation-card.unknown { border-left-color: #6c757d; }
        
        .status-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
            color: white;
        }
        .status-compatible { background-color: #28a745; }
        .status-fallback { background-color: #ffc107; }
        .status-invalid { background-color: #dc3545; }
        .status-unknown { background-color: #6c757d; }
        
        .validation-details {
            margin-top: 8px;
            font-size: 14px;
            color: #666;
        }
        
        .batch-actions {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            text-align: center;
        }
    </style>
</head>
<body>
    <h2 style="text-align:center;margin-bottom:20px;">Batch Template Validation</h2>
    
    <div style="text-align:center;margin-bottom:20px;">
        <a href="templates.php" class="btn" style="background:#6c757d;">← Back to Templates</a>
    </div>
    
    <?php if ($error): ?><p style="color:red; text-align:center;"><?= htmlspecialchars($error) ?></p><?php endif; ?>
    <?php if ($success): ?><p style="color:green; text-align:center;"><?= htmlspecialchars($success) ?></p><?php endif; ?>
    
    <div class="batch-actions">
        <h3>Batch Validation</h3>
        <p>Validate all active templates to check FPDI compatibility and identify potential issues.</p>
        <form method="post">
            <button type="submit" name="batch_validate" class="btn" style="background:#007bff;">
                Validate All Templates
            </button>
        </form>
    </div>
    
    <div class="user-table-section">
        <h3>Current Validation Status</h3>
        <div class="validation-summary">
            <?php foreach ($templates as $template): 
                $validationData = null;
                $statusClass = 'unknown';
                $statusText = 'Unknown';
                $statusTitle = 'Not validated';
                
                if (!empty($template['validation_data'])) {
                    $validationData = json_decode($template['validation_data'], true);
                    if ($validationData) {
                        if (!$validationData['is_valid']) {
                            $statusClass = 'invalid';
                            $statusText = 'Invalid';
                            $statusTitle = 'Template has validation errors';
                        } elseif ($validationData['fpdi_compatible'] === true) {
                            $statusClass = 'compatible';
                            $statusText = 'Compatible';
                            $statusTitle = 'Fully FPDI compatible';
                        } elseif ($validationData['fpdi_compatible'] === false) {
                            $statusClass = 'fallback';
                            $statusText = 'Fallback';
                            $statusTitle = 'Will use fallback generation';
                        }
                    }
                }
            ?>
            <div class="validation-card <?= $statusClass ?>">
                <h4><?= htmlspecialchars($template['name']) ?></h4>
                <div>
                    <span class="status-badge status-<?= $statusClass ?>" title="<?= htmlspecialchars($statusTitle) ?>">
                        <?= htmlspecialchars($statusText) ?>
                    </span>
                </div>
                
                <?php if ($validationData): ?>
                <div class="validation-details">
                    <?php if (!empty($validationData['validated_at'])): ?>
                        <div><strong>Validated:</strong> <?= htmlspecialchars($validationData['validated_at']) ?></div>
                    <?php endif; ?>
                    
                    <?php if (!empty($validationData['issues'])): ?>
                        <div><strong>Issues:</strong></div>
                        <ul style="margin:4px 0;padding-left:20px;">
                            <?php foreach ($validationData['issues'] as $issue): ?>
                                <li style="color:#dc3545;"><?= htmlspecialchars($issue) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    <?php endif; ?>
                    
                    <?php if (!empty($validationData['recommendations'])): ?>
                        <div><strong>Recommendations:</strong></div>
                        <ul style="margin:4px 0;padding-left:20px;">
                            <?php foreach ($validationData['recommendations'] as $rec): ?>
                                <li style="color:#856404;"><?= htmlspecialchars($rec) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    <?php endif; ?>
                    
                    <?php if (isset($validationData['test_duration_ms'])): ?>
                        <div><strong>Test Duration:</strong> <?= number_format($validationData['test_duration_ms'], 2) ?>ms</div>
                    <?php endif; ?>
                    
                    <?php if (isset($validationData['file_size_bytes'])): ?>
                        <div><strong>File Size:</strong> <?= number_format($validationData['file_size_bytes'] / 1024, 1) ?>KB</div>
                    <?php endif; ?>
                </div>
                <?php else: ?>
                <div class="validation-details">
                    <em>Template has not been validated yet.</em>
                </div>
                <?php endif; ?>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
    
    <?php if (!empty($results)): ?>
    <div class="user-table-section">
        <h3>Latest Batch Validation Results</h3>
        <div class="table-responsive">
            <table>
                <tr>
                    <th>Template</th>
                    <th>Status</th>
                    <th>FPDI Compatible</th>
                    <th>Test Duration</th>
                    <th>File Size</th>
                    <th>Issues</th>
                </tr>
                <?php foreach ($results as $result): ?>
                <tr>
                    <td><?= htmlspecialchars($result['template']['name']) ?></td>
                    <td>
                        <?php if ($result['validation']->isValid): ?>
                            <span style="color:#28a745;">✓ Valid</span>
                        <?php else: ?>
                            <span style="color:#dc3545;">✗ Invalid</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if ($result['validation']->fpdiCompatible === true): ?>
                            <span style="color:#28a745;">✓ Yes</span>
                        <?php elseif ($result['validation']->fpdiCompatible === false): ?>
                            <span style="color:#ffc107;">⚠ No</span>
                        <?php else: ?>
                            <span style="color:#6c757d;">? Unknown</span>
                        <?php endif; ?>
                    </td>
                    <td><?= number_format($result['quickTest']['testDuration'], 2) ?>ms</td>
                    <td><?= number_format($result['quickTest']['templateSize'] / 1024, 1) ?>KB</td>
                    <td>
                        <?php if (!empty($result['validation']->issues)): ?>
                            <ul style="margin:0;padding-left:16px;">
                                <?php foreach ($result['validation']->issues as $issue): ?>
                                    <li style="color:#dc3545;font-size:12px;"><?= htmlspecialchars($issue) ?></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php else: ?>
                            <span style="color:#28a745;">None</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
    </div>
    <?php endif; ?>
</body>
</html>