<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../vendor/autoload.php';
use App\Service\ServiceFactory;

$token = $_GET['token'] ?? null;
$license = '';
$error = '';
$success = '';
$fields = [];
$template = null;

if ($token) {
    $pdo = getDbConnection();
    $stmt = $pdo->prepare("SELECT lc.*, ct.field_definitions, ct.id as template_id, ct.file_path, ct.field_coordinates
        FROM license_codes lc
        JOIN certificate_templates ct ON lc.template_id = ct.id
        WHERE lc.token = ? LIMIT 1");
    $stmt->execute([$token]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        $error = "Invalid or expired link.";
    } elseif ($row['status'] == 'used') {
        $error = "This link has already been used.";
    } elseif (empty($row['download_created_at']) || strtotime($row['download_created_at']) === false) {
        $error = "This link is invalid or missing a creation date.";
    } elseif (strtotime($row['download_created_at']) < time() - (30 * 24 * 3600)) {
        $error = "This link has expired.";
    } else {
        $fields = json_decode($row['field_definitions'], true);
        $template = $row;
        $license = $row['code'];
    }
} else {
    $error = "No token provided.";
}

// Handle form submit
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $template) {
    $email = trim($_POST['email'] ?? '');
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = "A valid email address is required.";
    }
    $field_values = [];
    foreach ($fields as $field) {
        $field_values[$field] = trim($_POST[$field] ?? '');
        if ($field_values[$field] === '') {
            $error = "Please fill all fields.";
        }
    }

    // Store email in license_codes table if not already stored
    $email_inserted = false;
    if ($email && !$error) {
        $checkEmail = $pdo->prepare("SELECT email FROM license_codes WHERE id = ?");
        $checkEmail->execute([$template['id'] ?? $template['template_id']]);
        $existingEmail = $checkEmail->fetchColumn();
        if (empty($existingEmail)) {
            $updateEmail = $pdo->prepare("UPDATE license_codes SET email = ? WHERE id = ?");
            $updateEmail->execute([$email, $template['id'] ?? $template['template_id']]);
            $email_inserted = true;
        }
    }

    if (!$error) {
        // Ensure clean output for PDF generation and redirect
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        $pdf_template = __DIR__ . '/../templates/' . $template['file_path'];
        
        // Prepare field data for the PdfProcessor service
        $fieldData = [
            'field_values' => $field_values,
            'field_coordinates' => !empty($template['field_coordinates']) ? json_decode($template['field_coordinates'], true) : []
        ];
        
        // Disable error display and ensure logging goes to files only
        ini_set('display_errors', 0);
        ini_set('log_errors', 1);
        
        // Start output buffering to prevent any output during PDF processing
        ob_start();
        
        // Use universal PdfProcessor service with enhanced error handling
        $pdfProcessor = ServiceFactory::getPdfProcessor();
        $result = $pdfProcessor->processPdfTemplate($pdf_template, $fieldData);
        
        // Discard any output from PDF processing
        ob_end_clean();
        
        if (!$result->success) {
            $error = "Certificate generation failed: " . $result->errorMessage;
            
            // Log detailed error information to file only
            error_log("Protected certificate generation failed: " . $result->errorMessage . " (Template: " . $template['file_path'] . ")", 3, sys_get_temp_dir() . '/certificate_generation.log');
            
            // Add suggested actions if available
            if (isset($result->metadata['suggestedActions']) && is_array($result->metadata['suggestedActions'])) {
                $error .= "<br><br><strong>Suggestions:</strong><ul>";
                foreach ($result->metadata['suggestedActions'] as $action) {
                    $error .= "<li>" . htmlspecialchars($action) . "</li>";
                }
                $error .= "</ul>";
            } else {
                $error .= " Please try again or contact support.";
            }
        } else {
            $pdfContent = $result->pdfContent;
            
            // Log successful generation with processing method to file only
            $method = $result->metadata['processingMethod'] ?? $result->processingMethod ?? 'unknown';
            error_log("Protected certificate generated successfully: Method={$method}, Template={$template['file_path']}, Email={$email}", 3, sys_get_temp_dir() . '/certificate_generation.log');
            
            // Log any fallback usage with detailed information to file only
            if ($method !== 'fpdi' && isset($result->metadata['userNotification'])) {
                error_log("Certificate form fallback used: " . $result->metadata['userNotification'] . " (Method: {$method})", 3, sys_get_temp_dir() . '/certificate_generation.log');
            }

            // Build verified path: /storage/certificates/verified/{year}/{email}/{certificate_id}.pdf
            $year = date('Y');
            $safe_email = preg_replace('/[^a-zA-Z0-9._-]/', '_', strtolower($email));
            $cert_dir = __DIR__ . "/../storage/certificates/verified/$year/$safe_email/";
            if (!is_dir($cert_dir)) mkdir($cert_dir, 0777, true);
            $certificate_id = uniqid('cert_', true);
            $cert_filename = $certificate_id . '.pdf';
            $cert_path = $cert_dir . $cert_filename;
            $relative_file_path = "verified/$year/$safe_email/$cert_filename";
            
            // Save the PDF content from the service
            file_put_contents($cert_path, $pdfContent);

            // Save certificate record with email, template_id, file_path, license_code, created_at
            $stmt = $pdo->prepare("INSERT INTO certificates (template_id, email, file_path, license_code, download_created_at) VALUES (?, ?, ?, ?, NOW())");
            $stmt->execute([$template['template_id'], $email, $relative_file_path, $license]);

            // Mark license as used
            $stmt = $pdo->prepare("UPDATE license_codes SET status = 'used', used_at = NOW() WHERE id = ?");
            $stmt->execute([$template['id']]);
            
            // Log successful certificate creation for audit trail to file only
            error_log("Protected certificate created successfully: License={$license}, Email={$email}, Template={$template['template_id']}, Method={$method}, File={$relative_file_path}", 3, sys_get_temp_dir() . '/certificate_generation.log');

            // Set success message and download URL for JavaScript handling
            $success = "Certificate generated successfully!";
            $download_url = "download_certificate.php?license_code=" . urlencode($license);
        }
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Protected Certificate Generation</title>
    <link rel="stylesheet" href="style.css">
    <style>
        .protected-badge { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 0.9em;
            font-weight: bold;
            display: inline-block;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(102, 126, 234, 0.3);
        }
        .info-box {
            background: #f8f9fa;
            border-left: 4px solid #667eea;
            padding: 15px;
            margin: 20px 0;
            border-radius: 4px;
            font-size: 0.9em;
        }
        .info-box h4 {
            margin: 0 0 10px 0;
            color: #667eea;
        }
        .security-notice {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            border-radius: 4px;
            padding: 15px;
            margin: 20px 0;
            border-left: 4px solid #ffc107;
            font-size: 0.9em;
        }
    </style>
    <link rel="stylesheet" href="loader.css">
</head>
<body>
    <div style="display:flex;flex-direction:column;align-items:center;margin-top:36px;">
        <img src="../img/logo.png" alt="Logo" style="width:120px;margin-bottom:18px;">
        <div class="protected-badge">🛡️ Protected Certificate Generation</div>
        <div class="form-section" style="width:100%;max-width:500px;background:#fff;border-radius:10px;box-shadow:0 2px 12px #0001;padding:32px 28px 24px 28px;">
            <h2 style="text-align:center;">Fill Certificate Details</h2>
            
            <?php if ($fields && !$error): ?>
            <div class="security-notice">
                <strong>🔒 Note:</strong> This is a protected certificate generation form. The link will be valid only once.
            </div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div style="color:#d32f2f; text-align:center; margin-bottom:15px; padding:15px; background:#ffebee; border-radius:4px; border-left:4px solid #f44336;">
                    <strong>⚠️ Error:</strong> <?= $error ?>
                    <?php if (strpos($error, 'Certificate generation failed') !== false): ?>
                        <div style="margin-top:10px; padding:10px; background:#fff3e0; border-radius:4px; color:#e65100; font-size:0.9em;">
                            <strong>Universal PDF System:</strong> Our system attempted multiple processing methods to generate your certificate. If this error persists, please contact support with your license code.
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div id="successMessage" style="color:#2e7d32; text-align:center; margin-bottom:15px; padding:15px; background:#e8f5e8; border-radius:4px; border-left:4px solid #4caf50;">
                    <strong>✅ Success:</strong> <?= htmlspecialchars($success) ?>
                    <div style="margin-top:10px;">
                        <button id="downloadBtn" class="btn" style="background:#4caf50;color:#fff;padding:8px 16px;border-radius:4px;border:none;cursor:pointer;">
                            📄 Download Certificate
                        </button>
                    </div>
                    <div style="margin-top:8px; font-size:0.9em; color:#666;">
                        Your certificate has been generated and is ready for download.
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($fields && !$error && (!isset($success) || !$success)): ?>
            <form method="post" id="certificateForm">
                <label style="display:block;margin-bottom:10px;font-weight:500;">
                    Email: 
                    <input type="email" name="email" required style="width:100%;padding:8px;margin-top:4px;border:1px solid #bbb;border-radius:4px;font-size:1rem;box-sizing:border-box;">
                </label>
                <?php foreach ($fields as $field): ?>
                    <label style="display:block;margin-bottom:10px;font-weight:500;">
                        <?= htmlspecialchars(ucfirst(str_replace('_', ' ', $field))) ?>: 
                        <input type="text" name="<?= htmlspecialchars($field) ?>" required style="width:100%;padding:8px;margin-top:4px;border:1px solid #bbb;border-radius:4px;font-size:1rem;box-sizing:border-box;">
                    </label>
                <?php endforeach; ?>
                <div style="text-align:center;margin-top:18px;">
                    <button type="submit" class="btn" style="background:#667eea;color:#fff;border:none;padding:10px 20px;border-radius:4px;cursor:pointer;font-size:1rem;">
                        Generate Protected Certificate
                    </button>
                </div>
            </form>
            
            <div style="margin-top: 20px; padding: 15px; background: #e8f4fd; border-radius: 4px; font-size: 0.9em; text-align: center;">
                <strong>Processing Information:</strong> Your certificate will be generated using the most appropriate method for the template. All processing is secure and tracked.
            </div>
            <?php elseif (isset($success) && $success): ?>
            <div style="text-align: center; margin-top: 20px;">
                <p style="color:#666; margin-bottom: 15px;">Thank you! Your certificate has been generated successfully.</p>
            </div>
            <?php endif; ?>
        </div>
    </div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    var form = document.getElementById('certificateForm');
    var loader = document.getElementById('globalLoaderOverlay');
    
    // Handle form submission
    if (form) {
        form.addEventListener('submit', function() {
            // Show loading state
            var submitBtn = form.querySelector('button[type="submit"]');
            if (submitBtn) {
                submitBtn.innerHTML = 'Generating Certificate...';
                submitBtn.disabled = true;
                submitBtn.style.background = '#999';
            }
            
            // Show loader
            if (loader) {
                loader.classList.remove('hidden');
            }
        });
    }
    
    // Handle success state - hide loader and show success message
    var successMessage = document.getElementById('successMessage');
    if (successMessage) {
        // Hide loader
        if (loader) {
            loader.classList.add('hidden');
        }
        
        // Handle download button click
        var downloadBtn = document.getElementById('downloadBtn');
        if (downloadBtn) {
            downloadBtn.addEventListener('click', function() {
                <?php if (isset($download_url)): ?>
                // Create download link and trigger download
                var downloadUrl = '<?= htmlspecialchars($download_url) ?>';
                var a = document.createElement('a');
                a.href = downloadUrl;
                a.download = 'certificate.pdf';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                
                // Show success message
                setTimeout(function() {
                    var successDiv = successMessage.querySelector('strong');
                    if (successDiv) {
                        successDiv.innerHTML = '✅ Downloaded Successfully!';
                    }
                }, 500);
                <?php endif; ?>
            });
        }
    }
});
</script>

<div id="globalLoaderOverlay" class="hidden"><div class="loader-spinner"></div></div>
</body>

</html>
