<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../vendor/autoload.php';

use setasign\Fpdi\TcpdfFpdi;
use App\Service\ServiceFactory;

$error = '';
$success = '';
$field_values = [];
$template = null;
$pdo = getDbConnection();

// Fetch all active protected teacher templates
$stmt = $pdo->prepare("SELECT * FROM certificate_templates WHERE use_case_type = 'protected' AND status = 'active'");
$stmt->execute();
$templates = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Support direct link with ?template_id=XX
$default_template_id = isset($_GET['template_id']) ? intval($_GET['template_id']) : (isset($_POST['template_id']) ? intval($_POST['template_id']) : 0);

if ($_SERVER['REQUEST_METHOD'] === 'POST' || $default_template_id) {
    $template_id = $default_template_id;

    // Fetch selected active template
    $stmt = $pdo->prepare("SELECT * FROM certificate_templates WHERE id = ? AND use_case_type = 'protected' AND status = 'active'");
    $stmt->execute([$template_id]);
    $template = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$template) {
        $error = "Invalid template selected.";
    } else {
        $fields = json_decode($template['field_definitions'], true);

        // Collect field values
        foreach ($fields as $field) {
            $field_values[$field] = trim($_POST[$field] ?? '');
            if ($field_values[$field] === '') {
                $error = "Please fill all fields.";
            }
        }
    }

    // If no error, generate PDF using the universal PDF system
    if (!$error && $template && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $pdf_template = __DIR__ . '/../templates/' . $template['file_path'];
        
        // Prepare field data for the universal PDF processor
        $fieldData = [
            'field_values' => $field_values,
            'field_coordinates' => json_decode($template['field_coordinates'], true)
        ];

        // Use the universal PdfProcessor service
        $pdfProcessor = ServiceFactory::getPdfProcessor();
        $result = $pdfProcessor->processPdfTemplate($pdf_template, $fieldData);

        if ($result->success) {
            // Log successful generation for protected certificates
            error_log("Protected certificate generated successfully: Template ID {$template_id}, Method: " . ($result->metadata['processingMethod'] ?? 'unknown'));
            
            // Save PDF to temporary location for download
            $temp_id = uniqid('temp_cert_', true);
            $temp_dir = __DIR__ . '/../storage/temp/';
            if (!is_dir($temp_dir)) mkdir($temp_dir, 0777, true);
            $temp_file = $temp_dir . $temp_id . '.pdf';
            file_put_contents($temp_file, $result->pdfContent);
            
            // Set success flag for JavaScript handling
            $success = "Certificate generated successfully!";
            $download_id = $temp_id;
        } else {
            // Handle error from universal PDF system
            $error = $result->errorMessage;
            
            // Log additional error details if available
            if (isset($result->metadata['technicalMessage'])) {
                error_log("Protected Certificate PDF Generation Error: " . $result->metadata['technicalMessage']);
            }
            
            // Add suggested actions to error message if available
            if (isset($result->metadata['suggestedActions']) && is_array($result->metadata['suggestedActions'])) {
                $error .= "<br><br><strong>Suggestions:</strong><ul>";
                foreach ($result->metadata['suggestedActions'] as $action) {
                    $error .= "<li>" . htmlspecialchars($action) . "</li>";
                }
                $error .= "</ul>";
            }
            
            // Add processing method information for debugging
            if (isset($result->metadata['processingMethod'])) {
                error_log("Protected certificate failed with method: " . $result->metadata['processingMethod']);
            }
        }
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Protected/Verified Teacher Certificate Generator</title>
    <link rel="stylesheet" href="style.css">
    <style>
        .form-section { max-width: 500px; margin: 0 auto; background: #fff; border-radius: 8px; box-shadow: 0 2px 12px #0001; padding: 32px 28px 24px 28px; }
        .form-section label { display: block; margin-bottom: 10px; font-weight: 500; }
        .form-section input[type="text"],
        .form-section input[type="email"],
        .form-section select { width: 100%; padding: 8px; margin-top: 4px; margin-bottom: 2px; border: 1px solid #bbb; border-radius: 4px; font-size: 1rem; box-sizing: border-box; }
        .form-section .btn { margin-top: 10px; }
        .result-section { margin-top: 28px; text-align: center; }
        .result-section input[type=text] { width: 80%; max-width: 480px; padding: 8px; font-size: 1rem; }
        .protected-badge { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 0.9em;
            font-weight: bold;
            display: inline-block;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(102, 126, 234, 0.3);
        }
        .info-box {
            background: #f8f9fa;
            border-left: 4px solid #667eea;
            padding: 15px;
            margin: 20px 0;
            border-radius: 4px;
        }
        .info-box h4 {
            margin: 0 0 10px 0;
            color: #667eea;
        }
    </style>
    <link rel="stylesheet" href="loader.css">
</head>
<body>
<div style="display:flex;flex-direction:column;align-items:center;margin-top:36px;">
<img src="../img/logo.png" alt="Logo" style="width:120px;margin-bottom:18px;">
    <div class="protected-badge">🛡️ Protected/Verified Teacher Certificates</div>
    <h2 style="text-align:center;">Protected Teacher Certificate Generator</h2>
    
    <div class="info-box">
        <h4>Universal PDF Compatibility</h4>
        <p>This system accepts <strong>any PDF template</strong> and ensures certificate generation always succeeds using advanced fallback methods:</p>
        <ul style="margin: 10px 0; padding-left: 20px;">
            <li><strong>FPDI Method:</strong> Fast processing for compatible PDFs</li>
            <li><strong>Image Conversion:</strong> Preserves original template design for complex PDFs</li>
            <li><strong>Template Recreation:</strong> Professional certificates with intelligent positioning</li>
            <li><strong>Emergency Fallback:</strong> Guarantees success even with problematic templates</li>
        </ul>
    </div>
    
    <div class="form-section">
        <?php if ($error): ?><div style="color:red; text-align:center; margin-bottom: 20px;"><?= $error ?></div><?php endif; ?>
    
    <?php if (isset($success) && $success): ?>
        <div id="successMessage" style="color:#2e7d32; text-align:center; margin-bottom:20px; padding:15px; background:#e8f5e8; border-radius:4px; border-left:4px solid #4caf50;">
            <strong>✅ Success:</strong> <?= htmlspecialchars($success) ?>
            <div style="margin-top:10px;">
                <button id="downloadBtn" class="btn" style="background:#4caf50;color:#fff;">
                    📄 Download Certificate
                </button>
            </div>
            <div style="margin-top:8px; font-size:0.9em; color:#666;">
                Your protected certificate has been generated and is ready for download.
            </div>
        </div>
    <?php endif; ?>
        <?php if (!isset($success) || !$success): ?>
        <form method="post" id="certificateForm">
            <?php if (!$default_template_id): ?>
                <label>Select Protected Certificate Template:
                    <select name="template_id" required onchange="this.form.submit()">
                        <option value="">-- Select Template --</option>
                        <?php foreach ($templates as $tpl): ?>
                            <option value="<?= $tpl['id'] ?>" <?= (isset($_POST['template_id']) && $_POST['template_id'] == $tpl['id']) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($tpl['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label><br><br>
            <?php endif; ?>
            <?php
            // Show dynamic fields if template is selected
            if ($template) {
                $fields = json_decode($template['field_definitions'], true);
                echo "<div style='background: #e8f4fd; padding: 15px; border-radius: 4px; margin-bottom: 20px;'>";
                echo "<strong>Selected Template:</strong> " . htmlspecialchars($template['name']) . "<br>";
                echo "<strong>Template Type:</strong> Protected/Verified Teacher Certificate<br>";
                echo "<strong>Required Fields:</strong> " . implode(', ', $fields);
                echo "</div>";
                
                echo "<label>Email (optional, for record): <input type='email' name='email' value='" . htmlspecialchars($_POST['email'] ?? '') . "'></label>";
                foreach ($fields as $field) {
                    $val = htmlspecialchars($_POST[$field] ?? '');
                    echo "<label>" . ucfirst(str_replace('_', ' ', $field)) . ":<input type='text' name='" . htmlspecialchars($field) . "' value='$val' required></label>";
                }
            }
            ?>
            <button type="submit" class="btn">Generate Protected Certificate</button>
        </form>
        
        <?php if ($template): ?>
        <div style="margin-top: 20px; padding: 15px; background: #fff3cd; border-radius: 4px; border-left: 4px solid #ffc107;">
            <strong>🔒 Security Notice:</strong> This is a protected certificate template. Generated certificates are for verified/protected teacher credentials only.
        </div>
        
        <div style="margin-top: 20px; padding: 15px; background: #e8f4fd; border-radius: 4px; text-align: center;">
            <h4 style="margin: 0 0 10px 0; color: #0056b3;">📋 Bulk Generation Available</h4>
            <p style="margin: 0 0 15px 0;">Need to generate multiple protected certificates?</p>
            <a href="protected_bulk_generate.php?template_id=<?= $template['id'] ?>" class="btn" style="text-decoration: none; display: inline-block;">
                Generate Bulk Protected Certificates
            </a>
        </div>
        <?php endif; ?>
        <?php else: ?>
        <div style="text-align: center; margin-top: 20px;">
            <a href="protected_teacher_certificate.php" class="btn" style="background:#6c757d;color:#fff;text-decoration:none;padding:8px 16px;border-radius:4px;display:inline-block;">
                🔄 Generate Another Certificate
            </a>
        </div>
        <?php endif; ?>
    </div>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var form = document.getElementById('certificateForm');
    var loader = document.getElementById('globalLoaderOverlay');
    
    // Handle form submission
    if (form) {
        form.addEventListener('submit', function() {
            // Show loader
            if (loader) {
                loader.classList.remove('hidden');
            }
        });
    }
    
    // Handle success state - hide loader and enable download
    var successMessage = document.getElementById('successMessage');
    if (successMessage) {
        // Hide loader
        if (loader) {
            loader.classList.add('hidden');
        }
        
        // Handle download button
        var downloadBtn = document.getElementById('downloadBtn');
        if (downloadBtn) {
            downloadBtn.addEventListener('click', function() {
                <?php if (isset($download_id)): ?>
                // Create download link and trigger download
                var downloadUrl = 'temp_download.php?id=<?= htmlspecialchars($download_id) ?>';
                var a = document.createElement('a');
                a.href = downloadUrl;
                a.download = 'protected_certificate.pdf';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                
                // Show success message
                setTimeout(function() {
                    var successDiv = successMessage.querySelector('strong');
                    if (successDiv) {
                        successDiv.innerHTML = '✅ Downloaded Successfully!';
                    }
                }, 500);
                <?php endif; ?>
            });
        }
    }
});
</script>
<div id="globalLoaderOverlay" class="hidden"><div class="loader-spinner"></div></div>
</body>
</html>