<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../vendor/autoload.php';

use setasign\Fpdi\TcpdfFpdi;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use App\Service\ServiceFactory;

// Teacher bulk generation is accessible to all users (no admin restriction)

// Get template_id from URL
$template_id = isset($_GET['template_id']) ? intval($_GET['template_id']) : 0;
$pdo = getDbConnection();
$tpl_stmt = $pdo->prepare("SELECT * FROM certificate_templates WHERE id = ? AND use_case_type = 'general_teacher' AND status = 'active'");
$tpl_stmt->execute([$template_id]);
$template = $tpl_stmt->fetch(PDO::FETCH_ASSOC);

if (!$template) {
    die("Invalid or missing teacher template.");
}
$field_defs = json_decode($template['field_definitions'], true) ?: [];

// Handle Excel template download
if (isset($_GET['download_template'])) {
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    
    // Set headers (convert field names to a more readable format)
    $headers = [];
    foreach ($field_defs as $field) {
        $headers[] = ucwords(str_replace('_', ' ', $field));
    }
    
    // Add sample data row
    $sampleData = [];
    foreach ($field_defs as $field) {
        $sampleData[] = 'Sample ' . ucwords(str_replace('_', ' ', $field));
    }
    
    // Set data to spreadsheet
    $sheet->fromArray([$headers, $sampleData]);
    
    // Set headers for download
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="bulk_protected_teacher_template.xlsx"');
    header('Cache-Control: max-age=0');
    
    // Save to output
    $writer = new Xlsx($spreadsheet);
    $writer->save('php://output');
    exit;
}

$error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['teacher_file'])) {
    $file = $_FILES['teacher_file'];
    $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $tmp_path = $file['tmp_name'];

    // Parse file (CSV or Excel)
    $teachers = [];
    try {
        if ($file_ext === 'csv') {
            $handle = fopen($tmp_path, 'r');
            while (($data = fgetcsv($handle)) !== false) {
                $teachers[] = $data;
            }
            fclose($handle);
        } elseif (in_array($file_ext, ['xls', 'xlsx'])) {
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($tmp_path);
            $sheet = $spreadsheet->getActiveSheet();
            foreach ($sheet->getRowIterator() as $row) {
                $cellIterator = $row->getCellIterator();
                $cellIterator->setIterateOnlyExistingCells(false);
                $rowData = [];
                foreach ($cellIterator as $cell) {
                    $rowData[] = $cell->getValue();
                }
                $teachers[] = $rowData;
            }
        } else {
            $error = 'Unsupported file format.';
        }
    } catch (Exception $e) {
        $error = 'Error parsing file: ' . $e->getMessage();
    }

    // Generate PDFs and ZIP if no error
    if (!$error && count($teachers) > 1) {
        // Disable error display and start output buffering
        ini_set('display_errors', 0);
        ini_set('log_errors', 1);
        
        // Start output buffering to prevent any output during processing
        ob_start();
        $header = array_map(function($h) {
            return strtolower(str_replace([' ', '_'], '', $h));
        }, $teachers[0]);
        $pdf_paths = [];
        $cert_dir = sys_get_temp_dir() . '/tmp_protected_certs_' . uniqid();
        mkdir($cert_dir);
        
        // Track processing results
        $successful_certs = 0;
        $failed_certs = 0;
        $fallback_certs = 0;
        $processing_errors = [];
        
        for ($i = 1; $i < count($teachers); $i++) {
            $data = [];
            foreach ($teachers[$i] as $k => $v) {
                $data[$header[$k]] = $v;
            }

            // Get certificate recipient name for filename (prioritize certificate name fields)
            $certificate_name = $data['certificatename'] ?? $data['name'] ?? $data['teachername'] ?? 
                               $data['teacher'] ?? $data['recipientname'] ?? $data['recipient'] ?? 
                               $data['fullname'] ?? 'Teacher_Certificate_' . $i;
            
            // Create safe filename from certificate name
            $safe_filename = preg_replace('/[^a-zA-Z0-9_\-\s]/', '', $certificate_name);
            $safe_filename = preg_replace('/\s+/', '_', trim($safe_filename));
            $safe_filename = substr($safe_filename, 0, 50);
            if (empty($safe_filename)) $safe_filename = 'Teacher_Certificate_' . $i;
            
            // Generate PDF using universal PdfProcessor service
            $pdf_path = $cert_dir . '/' . $safe_filename . '.pdf';
            $pdf_template_path = __DIR__ . '/../templates/' . $template['file_path'];
            
            // Prepare field data for the universal PDF processor
            $fieldData = [
                'field_values' => [],
                'field_coordinates' => json_decode($template['field_coordinates'], true) ?: []
            ];
            
            // Map teacher data to field values
            foreach ($field_defs as $field) {
                $norm = strtolower(str_replace([' ', '_'], '', $field));
                $fieldData['field_values'][$field] = $data[$norm] ?? '';
            }
            
            // Use universal PdfProcessor service with retry mechanism
            $pdfProcessor = ServiceFactory::getPdfProcessor();
            $result = null;
            $max_retries = 2;
            $retry_count = 0;
            
            // Suppress any output during PDF processing
            ob_start();
            
            // Retry mechanism for failed certificates
            do {
                $result = $pdfProcessor->processPdfTemplate($pdf_template_path, $fieldData);
                
                if ($result->success) {
                    break; // Success, exit retry loop
                }
                
                $retry_count++;
                if ($retry_count < $max_retries) {
                    // Log retry attempt to file only
                    error_log("Retrying protected certificate generation for teacher {$i}, attempt {$retry_count}", 3, sys_get_temp_dir() . '/certificate_generation.log');
                    // Small delay before retry
                    usleep(100000); // 0.1 second delay
                }
            } while ($retry_count < $max_retries && !$result->success);
            
            // Discard any output from PDF processing
            ob_end_clean();
            
            if ($result->success) {
                // Save the generated PDF
                file_put_contents($pdf_path, $result->pdfContent);
                $pdf_paths[] = $pdf_path;
                $successful_certs++;
                
                // Track fallback usage
                $method = $result->metadata['processingMethod'] ?? $result->processingMethod ?? 'unknown';
                if ($method !== 'fpdi') {
                    $fallback_certs++;
                    // Log to file only, not to output
                    if (isset($result->metadata['userNotification'])) {
                        error_log("Protected teacher certificate {$i}: " . $result->metadata['userNotification'], 3, sys_get_temp_dir() . '/certificate_generation.log');
                    }
                }
            } else {
                // Handle PDF generation error
                $failed_certs++;
                $teacher_name = $certificate_name; // Use the same name as filename
                $processing_errors[] = [
                    'teacher' => $teacher_name,
                    'error' => $result->errorMessage,
                    'row' => $i,
                    'retries' => $retry_count
                ];
                
                error_log("Failed to generate protected certificate for teacher {$i} ({$teacher_name}): " . $result->errorMessage, 3, sys_get_temp_dir() . '/certificate_generation.log');
                
                // Create error PDF with teacher information
                $errorPdf = new TcpdfFpdi();
                $errorPdf->AddPage();
                $errorPdf->SetFont('helvetica', 'B', 16);
                $errorPdf->SetTextColor(200, 0, 0);
                $errorPdf->Cell(0, 20, 'Protected Certificate Generation Failed', 0, 1, 'C');
                $errorPdf->SetFont('helvetica', '', 12);
                $errorPdf->SetTextColor(0, 0, 0);
                $errorPdf->Ln(10);
                $errorPdf->Cell(0, 10, 'Teacher: ' . $teacher_name, 0, 1, 'L');
                $errorPdf->Cell(0, 10, 'Error: ' . $result->errorMessage, 0, 1, 'L');
                $errorPdf->Cell(0, 10, 'Template: ' . basename($pdf_template_path), 0, 1, 'L');
                $errorPdf->Cell(0, 10, 'Type: Protected/Verified Teacher Certificate', 0, 1, 'L');
                if (isset($result->metadata['suggestedActions'])) {
                    $errorPdf->Ln(5);
                    $errorPdf->Cell(0, 10, 'Suggested Actions:', 0, 1, 'L');
                    foreach ($result->metadata['suggestedActions'] as $action) {
                        $errorPdf->Cell(0, 8, '• ' . $action, 0, 1, 'L');
                    }
                }
                $errorPdf->Output($pdf_path, 'F');
                $pdf_paths[] = $pdf_path;
            }
        }
        
        // Create processing summary
        $total_teachers = count($teachers) - 1;
        $success_rate = $total_teachers > 0 ? round(($successful_certs / $total_teachers) * 100, 1) : 0;
        
        $summary_content = "Protected Teacher Certificate Generation Summary\n";
        $summary_content .= "==============================================\n\n";
        $summary_content .= "Total Teachers: {$total_teachers}\n";
        $summary_content .= "Successful: {$successful_certs}\n";
        $summary_content .= "Failed: {$failed_certs}\n";
        $summary_content .= "Used Fallback: {$fallback_certs}\n";
        $summary_content .= "Success Rate: {$success_rate}%\n\n";
        
        if (!empty($processing_errors)) {
            $summary_content .= "Errors:\n";
            $summary_content .= "--------\n";
            foreach ($processing_errors as $error) {
                $retry_info = $error['retries'] > 0 ? " (after {$error['retries']} retries)" : "";
                $summary_content .= "Row {$error['row']} - {$error['teacher']}: {$error['error']}{$retry_info}\n";
            }
            $summary_content .= "\n";
        }
        
        if ($failed_certs > 0 && $successful_certs > 0) {
            $summary_content .= "PARTIAL SUCCESS: Some protected certificates could not be generated.\n";
            $summary_content .= "This ZIP contains {$successful_certs} successful certificates and error reports for failed ones.\n\n";
        } elseif ($failed_certs > 0) {
            $summary_content .= "GENERATION FAILED: No protected certificates could be generated successfully.\n";
            $summary_content .= "This ZIP contains error reports for all failed attempts.\n\n";
        }
        
        if ($fallback_certs > 0) {
            $summary_content .= "Note: {$fallback_certs} certificates were generated using fallback methods due to template compatibility issues.\n";
            $summary_content .= "These certificates may have different formatting but contain all required information.\n\n";
        }
        
        $summary_content .= "Template: " . ($template['name'] ?? 'Unknown') . " (General Teacher)\n";
        $summary_content .= "Generated: " . date('Y-m-d H:i:s') . "\n";
        $summary_content .= "Type: General Teacher Certificates\n";
        
        // Save summary to file
        $summary_path = $cert_dir . '/processing_summary.txt';
        file_put_contents($summary_path, $summary_content);
        
        // Create ZIP with standard filename
        $zip_filename = 'certificates.zip';
        
        $zip_path = $cert_dir . '/' . $zip_filename;
        $zip = new ZipArchive();
        $zip_result = $zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE);
        if ($zip_result === TRUE) {
            // Add summary first
            if (file_exists($summary_path)) {
                $zip->addFile($summary_path, 'processing_summary.txt');
            }
            
            // Add all PDFs (including error PDFs)
            foreach ($pdf_paths as $pdf) {
                if (file_exists($pdf)) {
                    $zip->addFile($pdf, basename($pdf));
                }
            }
            $zip->close();
            
            // Verify ZIP was created successfully
            if (!file_exists($zip_path) || filesize($zip_path) == 0) {
                $error = "Failed to create ZIP file. Please try again.";
            }
        } else {
            $error = "Failed to create ZIP archive. Error code: " . $zip_result;
        }
        
        // Only proceed with download if no errors occurred
        if (!$error) {
            // Move ZIP to temporary download location
            $temp_id = uniqid('bulk_zip_', true);
            $temp_dir = __DIR__ . '/../storage/temp/';
            if (!is_dir($temp_dir)) mkdir($temp_dir, 0777, true);
            $temp_zip_path = $temp_dir . $temp_id . '.zip';
            
            // Move the ZIP file to temp location
            if (rename($zip_path, $temp_zip_path)) {
                // Set success flag for JavaScript handling
                $success = "Bulk teacher certificates generated successfully!";
                $download_id = $temp_id;
                
                // Cleanup PDFs and summary (but keep the ZIP in temp)
                foreach ($pdf_paths as $pdf) { 
                    if (file_exists($pdf)) unlink($pdf); 
                }
                if (file_exists($summary_path)) unlink($summary_path);
                if (is_dir($cert_dir) && count(scandir($cert_dir)) == 2) rmdir($cert_dir);
            } else {
                $error = "Failed to prepare download. Please try again.";
            }
        }
    } elseif (!$error && count($teachers) <= 1) {
        $error = 'No teacher records found in the uploaded file.';
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>Bulk Teacher Certificate Upload</title>
    <style>
        body { font-family: Arial, sans-serif; }
        .container { max-width: 500px; margin: 40px auto; padding: 32px; background: #fff; border-radius: 10px; box-shadow: 0 2px 12px rgba(0,0,0,0.08);}
        .btn { background: #667eea; color: #fff; border: none; padding: 8px 18px; border-radius: 4px; cursor: pointer;}
        .btn:hover { background: #5a67d8; }
        .error { color: #b00; margin-bottom: 10px; }
        .protected-badge { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 0.9em;
            font-weight: bold;
            display: inline-block;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(102, 126, 234, 0.3);
        }
        .security-notice {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            border-radius: 4px;
            padding: 15px;
            margin: 20px 0;
            border-left: 4px solid #ffc107;
        }
        .custom-file-input { display: none; }
        .custom-file-label {
            display: inline-block;
            background: #667eea;
            color: #fff;
            padding: 8px 18px;
            border-radius: 4px;
            cursor: pointer;
            margin-bottom: 8px;
            transition: background 0.2s;
        }
        .custom-file-label:hover { background: #5a67d8; }
        #file-name {
            margin-left: 10px;
            font-size: 0.96em;
            color: #333;
        }
    </style>
    <link rel="stylesheet" href="loader.css">
</head>
<body>
<div style="display:flex;flex-direction:column;align-items:center;margin-top:36px;">
<img src="../img/logo.png" alt="Logo" style="max-width:180px;display:block;margin:0 auto 18px auto;">
<div class="protected-badge" style="background: linear-gradient(135deg, #28a745 0%, #20c997 100%);">👨‍🏫 Teacher Certificates</div>
<h2 style="text-align:center;margin-bottom:20px;">Generate Teacher Certificates - Bulk</h2>
<div class="container">
    <?php if ($error): ?>
        <div class="error">
            <strong>Error:</strong> <?= htmlspecialchars($error) ?>
            <?php if (strpos($error, 'PDF') !== false || strpos($error, 'template') !== false): ?>
                <br><small>If this is a template-related error, the system will attempt to use fallback generation methods.</small>
            <?php endif; ?>
        </div>
    <?php endif; ?>
    
    <?php if (isset($success) && $success): ?>
        <div id="successMessage" style="color:#2e7d32; text-align:center; margin-bottom:20px; padding:15px; background:#e8f5e8; border-radius:4px; border-left:4px solid #4caf50;">
            <strong>✅ Success:</strong> <?= htmlspecialchars($success) ?>
            <div style="margin-top:10px;">
                <button id="downloadBtn" class="btn" style="background:#4caf50;color:#fff;padding:8px 16px;border-radius:4px;border:none;cursor:pointer;">
                    📦 Download Certificates ZIP
                </button>
            </div>
            <div style="margin-top:8px; font-size:0.9em; color:#666;">
                Your teacher certificates have been generated and are ready for download.
            </div>
        </div>
    <?php endif; ?>
    <?php if (!isset($success) || !$success): ?>
    <div class="security-notice" style="background: #e8f5e9; border-left: 4px solid #28a745;">
        <strong>👨‍🏫 Teacher Certificates:</strong> Upload your CSV/Excel file with teacher information to begin.
    </div>
    
    <div style="margin-bottom: 20px; padding: 15px; background: #f8f9fa; border-radius: 4px; text-align: center;">
        <a href="?template_id=<?= $template_id ?>&download_template=1" style="text-decoration: none; color: #333;">
            <img src="../img/download.png" alt="Download" style="vertical-align: middle; height: 20px; margin-right: 8px;">
            <span style="vertical-align: middle; font-weight: bold;">Download the Excel template</span>
        </a>
        <div style="margin-top: 8px; color: #6c757d; font-size: 0.9em;">
            Contains all required column headers for teacher certificates
        </div>
    </div>
    
    <form method="post" enctype="multipart/form-data" style="margin-top: 20px;" id="bulkForm">
        <p><strong>Required columns in your CSV/Excel:</strong> <?= implode(', ', $field_defs) ?></p>
        <div style="display: flex; align-items: center; gap: 10px; margin: 10px 0;">
            <label class="custom-file-label">
                Choose File
                <input type="file" name="teacher_file" id="teacher_file" class="custom-file-input" accept=".csv,.xlsx,.xls" required>
            </label>
            <span id="file-name" style="color: #666;">No file chosen</span>
        </div>
        <br><br>
        <button type="submit" class="btn">Generate Teacher Certificates</button>
    </form>

    
   
    <?php endif; ?>
</div>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var fileInput = document.getElementById('teacher_file');
    var form = document.getElementById('bulkForm');
    var loader = document.getElementById('globalLoaderOverlay');
    
    // File selection handler
    if (fileInput) {
        fileInput.addEventListener('change', function(){
            var fileName = this.files[0] ? this.files[0].name : 'No file chosen';
            document.getElementById('file-name').textContent = fileName;
        });
    }
    
    // Form submission handler
    if (form) {
        form.addEventListener('submit', function() {
            // Show loader
            if (loader) {
                loader.classList.remove('hidden');
            }
        });
    }
    
    // Handle success state - hide loader and enable download
    var successMessage = document.getElementById('successMessage');
    if (successMessage) {
        // Hide loader
        if (loader) {
            loader.classList.add('hidden');
        }
        
        // Handle download button
        var downloadBtn = document.getElementById('downloadBtn');
        if (downloadBtn) {
            downloadBtn.addEventListener('click', function() {
                <?php if (isset($download_id)): ?>
                // Create download link and trigger download
                var downloadUrl = 'bulk_download.php?file=<?= htmlspecialchars($download_id) ?>&type=zip';
                var a = document.createElement('a');
                a.href = downloadUrl;
                a.download = 'certificates.zip';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                
                // Show success message
                setTimeout(function() {
                    var successDiv = successMessage.querySelector('strong');
                    if (successDiv) {
                        successDiv.innerHTML = '✅ Downloaded Successfully!';
                    }
                }, 500);
                <?php endif; ?>
            });
        }
    }
});
</script>
<div id="globalLoaderOverlay" class="hidden"><div class="loader-spinner"></div></div>
</body>
</html>