<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../vendor/autoload.php';

use setasign\Fpdi\TcpdfFpdi;
use App\Service\ServiceFactory;

$error = '';
$success = '';
$field_values = [];
$template = null;
$pdo = getDbConnection();

// Fetch all active general teacher templates
$stmt = $pdo->prepare("SELECT * FROM certificate_templates WHERE use_case_type = 'general_teacher' AND status = 'active'");
$stmt->execute();
$templates = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Support direct link with ?template_id=XX
$default_template_id = isset($_GET['template_id']) ? intval($_GET['template_id']) : (isset($_POST['template_id']) ? intval($_POST['template_id']) : 0);

if ($_SERVER['REQUEST_METHOD'] === 'POST' || $default_template_id) {
    $template_id = $default_template_id;

    // Fetch selected active template
    $stmt = $pdo->prepare("SELECT * FROM certificate_templates WHERE id = ? AND use_case_type = 'general_teacher' AND status = 'active'");
    $stmt->execute([$template_id]);
    $template = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$template) {
        $error = "Invalid template selected.";
    } else {
        $fields = json_decode($template['field_definitions'], true);

        // Collect field values
        foreach ($fields as $field) {
            $field_values[$field] = trim($_POST[$field] ?? '');
            if ($field_values[$field] === '') {
                $error = "Please fill all fields.";
            }
        }
    }

    // If no error, generate PDF
    if (!$error && $template && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $pdf_template = __DIR__ . '/../templates/' . $template['file_path'];
        
        // Prepare field data for the service
        $fieldData = [
            'field_values' => $field_values,
            'field_coordinates' => json_decode($template['field_coordinates'], true)
        ];

        // Use the new PdfProcessor service
        $pdfProcessor = ServiceFactory::getPdfProcessor();
        $result = $pdfProcessor->processPdfTemplate($pdf_template, $fieldData);

        if ($result->success) {
            // Save PDF to temporary location for download
            $temp_id = uniqid('temp_cert_', true);
            $temp_dir = __DIR__ . '/../storage/temp/';
            if (!is_dir($temp_dir)) mkdir($temp_dir, 0777, true);
            $temp_file = $temp_dir . $temp_id . '.pdf';
            file_put_contents($temp_file, $result->pdfContent);
            
            // Set success flag for JavaScript handling
            $success = "Certificate generated successfully!";
            $download_id = $temp_id;
        } else {
            // Handle error from service
            $error = $result->errorMessage;
            
            // Log additional error details if available
            if (isset($result->metadata['technicalMessage'])) {
                error_log("PDF Generation Error: " . $result->metadata['technicalMessage']);
            }
            
            // Add suggested actions to error message if available
            if (isset($result->metadata['suggestedActions']) && is_array($result->metadata['suggestedActions'])) {
                $error .= "<br><br><strong>Suggestions:</strong><ul>";
                foreach ($result->metadata['suggestedActions'] as $action) {
                    $error .= "<li>" . htmlspecialchars($action) . "</li>";
                }
                $error .= "</ul>";
            }
        }
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Teacher Certificate Generator</title>
    <link rel="stylesheet" href="style.css">
    <style>
        .form-section { max-width: 500px; margin: 0 auto; background: #fff; border-radius: 8px; box-shadow: 0 2px 12px #0001; padding: 32px 28px 24px 28px; }
        .form-section label { display: block; margin-bottom: 10px; font-weight: 500; }
        .form-section input[type="text"],
        .form-section select { width: 100%; padding: 8px; margin-top: 4px; margin-bottom: 2px; border: 1px solid #bbb; border-radius: 4px; font-size: 1rem; box-sizing: border-box; }
        .form-section .btn { margin-top: 10px; }
        .result-section { margin-top: 28px; text-align: center; }
        .result-section input[type=text] { width: 80%; max-width: 480px; padding: 8px; font-size: 1rem; }
    </style>
    <link rel="stylesheet" href="loader.css">
</head>
<body>
<div style="display:flex;flex-direction:column;align-items:center;margin-top:36px;">
<img  src="../img/logo.png" alt="Logo" style="width:120px;margin-bottom:18px;">
    <h2 style="text-align:center;">Teacher Certificate Generator</h2>
    <div class="form-section">
        <?php if ($error): ?><div style="color:red; text-align:center; margin-bottom: 20px;"><?= $error ?></div><?php endif; ?>
        
        <?php if (isset($success) && $success): ?>
            <div id="successMessage" style="color:#2e7d32; text-align:center; margin-bottom:20px; padding:15px; background:#e8f5e8; border-radius:4px; border-left:4px solid #4caf50;">
                <strong>✅ Success:</strong> <?= htmlspecialchars($success) ?>
                <div style="margin-top:10px;">
                    <button id="downloadBtn" class="btn" style="background:#4caf50;color:#fff;">
                        📄 Download Certificate
                    </button>
                </div>
                <div style="margin-top:8px; font-size:0.9em; color:#666;">
                    Your certificate has been generated and is ready for download.
                </div>
            </div>
        <?php endif; ?>
        
        <?php if (!isset($success) || !$success): ?>
        <form method="post" id="certificateForm">
            <?php if (!$default_template_id): ?>
                <label>Select Certificate Template:
                    <select name="template_id" required onchange="this.form.submit()">
                        <option value="">-- Select --</option>
                        <?php foreach ($templates as $tpl): ?>
                            <option value="<?= $tpl['id'] ?>" <?= (isset($_POST['template_id']) && $_POST['template_id'] == $tpl['id']) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($tpl['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label><br><br>
            <?php endif; ?>
            <?php
            // Show dynamic fields if template is selected
            if ($template) {
                $fields = json_decode($template['field_definitions'], true);
                foreach ($fields as $field) {
                    $val = htmlspecialchars($_POST[$field] ?? '');
                    echo "<label>$field:<input type='text' name='" . htmlspecialchars($field) . "' value='$val' required></label>";
                }
            }
            ?>
            <button type="submit" class="btn">Download Certificate</button>
        </form>
        <?php endif; ?>
    </div>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var form = document.getElementById('certificateForm');
    var loader = document.getElementById('globalLoaderOverlay');
    
    // Handle form submission
    if (form) {
        form.addEventListener('submit', function() {
            // Show loader
            if (loader) {
                loader.classList.remove('hidden');
            }
        });
    }
    
    // Handle success state - hide loader and enable download
    var successMessage = document.getElementById('successMessage');
    if (successMessage) {
        // Hide loader
        if (loader) {
            loader.classList.add('hidden');
        }
        
        // Handle download button
        var downloadBtn = document.getElementById('downloadBtn');
        if (downloadBtn) {
            downloadBtn.addEventListener('click', function() {
                <?php if (isset($download_id)): ?>
                // Create download link and trigger download
                var downloadUrl = 'temp_download.php?id=<?= htmlspecialchars($download_id) ?>';
                var a = document.createElement('a');
                a.href = downloadUrl;
                a.download = 'certificate.pdf';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                
                // Show success message
                setTimeout(function() {
                    var successDiv = successMessage.querySelector('strong');
                    if (successDiv) {
                        successDiv.innerHTML = '✅ Downloaded Successfully!';
                    }
                }, 500);
                <?php endif; ?>
            });
        }
    }
});
</script>
<div id="globalLoaderOverlay" class="hidden"><div class="loader-spinner"></div></div>
</body>
</html>