<?php
session_start();
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../config/config.php';

use App\Service\ServiceFactory;

if (empty($_SESSION['user_logged_in']) && empty($_SESSION['admin_logged_in'])) {
    echo "<script>window.top.location.href = 'login.php';</script>";
    exit;
}

$pdo = getDbConnection();
$error = '';
$success = '';
$edit_tpl = null;

// Fetch template for editing
if (isset($_GET['edit'])) {
    $edit_id = intval($_GET['edit']);
    $stmt = $pdo->prepare("SELECT * FROM certificate_templates WHERE id = ?");
    $stmt->execute([$edit_id]);
    $edit_tpl = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Handle upload new template
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upload_template'])) {
    $name = trim($_POST['name']);
    $useCase = $_POST['use_case_type'];
    $fields = array_map('trim', explode(',', $_POST['fields']));
    $fieldCoordinates = $_POST['field_coordinates'] ?? '';
    $pdfPath = '';
    $validationWarnings = [];

    if (empty($name) || empty($useCase) || empty($fields) || empty($_FILES['pdf']['name'])) {
        $error = "All fields are required.";
    } else {
        $ext = strtolower(pathinfo($_FILES['pdf']['name'], PATHINFO_EXTENSION));
        if ($ext !== 'pdf') {
            $error = "File must be a PDF.";
        } else {
            $pdfName = uniqid('tpl_', true) . '.pdf';
            $destPath = __DIR__ . '/../templates/' . $pdfName;
            if (move_uploaded_file($_FILES['pdf']['tmp_name'], $destPath)) {
                // Validate the uploaded PDF template
                try {
                    $templateValidator = ServiceFactory::getTemplateValidator();
                    $validationResult = $templateValidator->validateTemplate($destPath);
                    
                    if (!$validationResult->isValid) {
                        // Template is invalid, remove the uploaded file and show error
                        unlink($destPath);
                        $error = "Template validation failed: " . implode(', ', $validationResult->issues);
                        if (!empty($validationResult->recommendations)) {
                            $error .= " Recommendations: " . implode(', ', $validationResult->recommendations);
                        }
                    } else {
                        // Template is valid - show processing method info
                        if (!$validationResult->fpdiCompatible) {
                            $validationWarnings[] = "Info: This PDF will be processed using our universal PDF handler for maximum compatibility.";
                            if (!empty($validationResult->recommendations)) {
                                $validationWarnings = array_merge($validationWarnings, $validationResult->recommendations);
                            }
                        } else {
                            $validationWarnings[] = "Info: This PDF is fully compatible and will be processed using the fastest method.";
                        }
                        
                        $pdfPath = $pdfName;
                        
                        // Store validation results in database
                        $validationData = json_encode([
                            'is_valid' => $validationResult->isValid,
                            'fpdi_compatible' => $validationResult->fpdiCompatible,
                            'issues' => $validationResult->issues,
                            'recommendations' => $validationResult->recommendations,
                            'validated_at' => date('Y-m-d H:i:s')
                        ]);
                        
                        if ($edit_tpl) {
                            $stmt = $pdo->prepare("UPDATE certificate_templates SET name = ?, use_case_type = ?, field_definitions = ?, field_coordinates = ?, file_path = ?, validation_data = ? WHERE id = ?");
                            $stmt->execute([$name, $useCase, json_encode($fields), $fieldCoordinates, $pdfPath, $validationData, $edit_tpl['id']]);
                            $success = "Template updated successfully.";
                            if (!empty($validationWarnings)) {
                                $success .= " " . implode(' ', $validationWarnings);
                            }
                            unset($edit_tpl);
                            header("Location: templates.php?updated=1");
                            exit;
                        } else {
                            $stmt = $pdo->prepare("INSERT INTO certificate_templates (name, use_case_type, field_definitions, field_coordinates, file_path, validation_data, created_by) VALUES (?, ?, ?, ?, ?, ?, ?)");
                            $stmt->execute([$name, $useCase, json_encode($fields), $fieldCoordinates, $pdfPath, $validationData, $_SESSION['admin_email']]);
                            $success = "Template uploaded successfully.";
                            if (!empty($validationWarnings)) {
                                $success .= " " . implode(' ', $validationWarnings);
                            }
                        }
                    }
                } catch (Exception $e) {
                    // If validation fails due to an error, log it but don't block upload
                    error_log("Template validation error: " . $e->getMessage());
                    $validationWarnings[] = "Info: Template uploaded successfully. Our universal PDF system supports all PDF formats.";
                    
                    $pdfPath = $pdfName;
                    if ($edit_tpl) {
                        $stmt = $pdo->prepare("UPDATE certificate_templates SET name = ?, use_case_type = ?, field_definitions = ?, field_coordinates = ?, file_path = ? WHERE id = ?");
                        $stmt->execute([$name, $useCase, json_encode($fields), $fieldCoordinates, $pdfPath, $edit_tpl['id']]);
                        $success = "Template updated successfully. " . implode(' ', $validationWarnings);
                        unset($edit_tpl);
                        header("Location: templates.php?updated=1");
                        exit;
                    } else {
                        $stmt = $pdo->prepare("INSERT INTO certificate_templates (name, use_case_type, field_definitions, field_coordinates, file_path, created_by) VALUES (?, ?, ?, ?, ?, ?)");
                        $stmt->execute([$name, $useCase, json_encode($fields), $fieldCoordinates, $pdfPath, $_SESSION['admin_email']]);
                        $success = "Template uploaded successfully. " . implode(' ', $validationWarnings);
                    }
                }
            } else {
                $error = "Failed to upload PDF.";
            }
        }
    }
}

// Handle download sample PDF
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'download_sample') {
    try {
        $fieldCoordinates = json_decode($_POST['field_coordinates'], true);
        $fieldValues = json_decode($_POST['field_values'], true);
        
        // Get template path
        $templatePath = '';
        if (isset($_POST['template_id'])) {
            // Existing template
            $templateId = intval($_POST['template_id']);
            $stmt = $pdo->prepare("SELECT file_path FROM certificate_templates WHERE id = ? AND status = 'active'");
            $stmt->execute([$templateId]);
            $template = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($template) {
                $templatePath = __DIR__ . '/../templates/' . $template['file_path'];
            }
        } elseif (isset($_FILES['pdf_file'])) {
            // New template - use uploaded file
            $uploadedFile = $_FILES['pdf_file'];
            if ($uploadedFile['error'] === UPLOAD_ERR_OK) {
                $templatePath = $uploadedFile['tmp_name'];
            }
        }
        
        if (empty($templatePath) || !file_exists($templatePath)) {
            throw new Exception('Template file not found');
        }
        
        // Prepare field data for PDF processor
        $fieldData = [
            'field_values' => $fieldValues,
            'field_coordinates' => $fieldCoordinates
        ];
        
        // Generate PDF using the same service as certificate generation
        $pdfProcessor = ServiceFactory::getPdfProcessor();
        $result = $pdfProcessor->processPdfTemplate($templatePath, $fieldData);
        
        if (!$result->success) {
            throw new Exception('PDF generation failed: ' . $result->errorMessage);
        }
        
        // Send PDF as download
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="sample-certificate.pdf"');
        header('Content-Length: ' . strlen($result->pdfContent));
        header('Cache-Control: private, max-age=0, must-revalidate');
        header('Pragma: public');
        
        echo $result->pdfContent;
        exit;
        
    } catch (Exception $e) {
        error_log("Sample PDF generation error: " . $e->getMessage());
        http_response_code(500);
        echo "Error generating PDF: " . $e->getMessage();
        exit;
    }
}

// Handle edit (not full interactive for brevity)
if (isset($_GET['edit'])) {
    $id = intval($_GET['edit']);
    $stmt = $pdo->prepare("SELECT * FROM certificate_templates WHERE id = ?");
    $stmt->execute([$id]);
    $edit_tpl = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$edit_tpl) $error = "Template not found.";
}

// Handle soft delete
if (isset($_GET['delete'])) {
    $id = intval($_GET['delete']);
    // Soft delete the template
    $stmt = $pdo->prepare("UPDATE certificate_templates SET status = 'deleted' WHERE id = ?");
    $stmt->execute([$id]);
    $success = "Template deleted (soft delete).";
    // Optionally, also soft delete related certificates/license codes
    // $stmt = $pdo->prepare("UPDATE certificates SET status = 'deleted' WHERE template_id = ?");
    // $stmt->execute([$id]);
    // $stmt = $pdo->prepare("UPDATE license_codes SET status = 'deleted' WHERE template_id = ?");
    // $stmt->execute([$id]);
}



// Handle template validation
if (isset($_GET['validate'])) {
    $id = intval($_GET['validate']);
    $stmt = $pdo->prepare("SELECT * FROM certificate_templates WHERE id = ? AND status = 'active'");
    $stmt->execute([$id]);
    $template = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($template) {
        $templatePath = __DIR__ . '/../templates/' . $template['file_path'];
        
        try {
            $templateValidator = ServiceFactory::getTemplateValidator();
            $validationResult = $templateValidator->validateTemplate($templatePath);
            
            // Store validation results
            $validationData = json_encode([
                'is_valid' => $validationResult->isValid,
                'fpdi_compatible' => $validationResult->fpdiCompatible,
                'issues' => $validationResult->issues,
                'recommendations' => $validationResult->recommendations,
                'validated_at' => date('Y-m-d H:i:s')
            ]);
            
            $stmt = $pdo->prepare("UPDATE certificate_templates SET validation_data = ? WHERE id = ?");
            $stmt->execute([$validationData, $id]);
            
            if ($validationResult->isValid) {
                if ($validationResult->fpdiCompatible) {
                    $success = "Template validation completed. Template is fully compatible.";
                } else {
                    $success = "Template validation completed. Template is valid but will use fallback generation due to compression compatibility issues.";
                }
            } else {
                $error = "Template validation failed: " . implode(', ', $validationResult->issues);
            }
            
        } catch (Exception $e) {
            error_log("Template validation error for ID {$id}: " . $e->getMessage());
            $error = "Validation failed due to an error. Please check the template file.";
        }
    } else {
        $error = "Template not found.";
    }
}
// Handle batch validation
if (isset($_GET['batch_validate'])) {
    try {
        $templateValidator = ServiceFactory::getTemplateValidator();
        $errorHandler = ServiceFactory::getErrorHandler();
        $managementService = new \App\Service\TemplateManagementService($templateValidator, $errorHandler);
        
        $batchResults = $managementService->batchValidateTemplates();
        
        // Update database with validation results
        foreach ($batchResults['templates'] as $templateId => $templateInfo) {
            $validationData = json_encode([
                'is_valid' => $templateInfo['is_valid'],
                'fpdi_compatible' => $templateInfo['fpdi_compatible'],
                'issues' => $templateInfo['issues'] ?? [],
                'recommendations' => $templateInfo['recommendations'] ?? [],
                'validated_at' => date('Y-m-d H:i:s')
            ]);
            
            // Find template by filename and update
            $stmt = $pdo->prepare("UPDATE certificate_templates SET validation_data = ? WHERE file_path = ? AND status = 'active'");
            $stmt->execute([$validationData, $templateInfo['filename']]);
        }
        
        $success = "Batch validation completed. {$batchResults['total_templates']} templates processed. {$batchResults['valid_templates']} valid, {$batchResults['fpdi_compatible']} FPDI compatible.";
        
    } catch (Exception $e) {
        error_log("Batch validation error: " . $e->getMessage());
        $error = "Batch validation failed: " . $e->getMessage();
    }
}

// Get template management data
$templateManagementData = null;
if (isset($_GET['show_management']) || isset($_GET['batch_validate'])) {
    try {
        $templateValidator = ServiceFactory::getTemplateValidator();
        $errorHandler = ServiceFactory::getErrorHandler();
        $managementService = new \App\Service\TemplateManagementService($templateValidator, $errorHandler);
        
        $templateManagementData = [
            'compatibility_status' => $managementService->getTemplateCompatibilityStatus(),
            'problematic_templates' => $managementService->getProblematicTemplates(),
            'recommendations' => $managementService->generateReplacementRecommendations(),
            'usage_report' => $managementService->generateUsageReport(30)
        ];
    } catch (Exception $e) {
        error_log("Template management data error: " . $e->getMessage());
    }
}

// Fetch all templates
$stmt = $pdo->query("SELECT * FROM certificate_templates WHERE status = 'active' ORDER BY id DESC");
$templates = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html>
<head>
    <title>Certificate Templates - CertifyEd</title>
    <link rel="stylesheet" href="style.css">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js"></script>
    <style>
        .user-table-section {
            width: 100%;
            margin: 0;
            padding: 0 24px 24px 24px;
            box-sizing: border-box;
            background: #fff;
            border-radius: 10px;
            box-shadow: 0 2px 12px #0001;
        }
        .user-table-section h3 {
            margin-top: 0;
            padding-top: 16px;
            font-size: 1.2rem;
            font-weight: 600;
            color: #222;
        }
        .user-table-section .table-responsive {
            overflow-x: auto;
        }
        .user-table-section table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
            background: #fff;
            font-size: 1rem;
            border-radius: 8px;
            overflow: hidden;
            min-width: 700px;
        }
        .user-table-section th, .user-table-section td {
            padding: 14px 18px;
            border-bottom: 1px solid #f0f0f0;
            text-align: left;
            vertical-align: middle;
            white-space: nowrap;
        }
        .user-table-section th {
            background: #f8fafc;
            font-weight: 700;
            color: #2d3748;
            border-bottom: 2px solid #e2e8f0;
        }
        .user-table-section tr:nth-child(even) td {
            background: #f6f8fa;
        }
        .user-table-section tr:hover td {
            background: #eaf3fb;
        }
        .user-table-section tr:last-child td {
            border-bottom: none;
        }
        .user-table-section td {
            word-break: break-word;
            white-space: normal;
        }
        @media (max-width: 900px) {
            .user-table-section table, .user-table-section th, .user-table-section td {
                font-size: 0.92rem;
            }
        }
    </style>
    <link rel="stylesheet" href="loader.css">
</head>
<body>
    <!-- Add this container div around your upload form section -->
    <h2 style="text-align:center;margin-bottom:20px;">Certificate Templates</h2>
   
    <?php
// Show success message after update redirect
if (isset($_GET['updated']) && $_GET['updated'] == '1') {
    $success = "Template updated successfully.";
}
?>

    <?php if ($error): ?><p style="color:red; text-align:center;"><?= htmlspecialchars($error) ?></p><?php endif; ?>
    <?php if ($success): ?><p style="color:green; text-align:center;"><?= htmlspecialchars($success) ?></p><?php endif; ?>

    <div class="user-form-section">
    <h3>Upload New Template</h3>
    <form method="post" enctype="multipart/form-data">
    <input type="hidden" name="upload_template" value="1">
    <?php if ($edit_tpl): ?>
        <input type="hidden" name="edit_id" value="<?= htmlspecialchars($edit_tpl['id']) ?>">
    <?php endif; ?>
    <label>Name: <input type="text" name="name" value="<?= htmlspecialchars($edit_tpl['name'] ?? '') ?>" required></label><br>
    <label>Use Case Type:
    <select name="use_case_type" required>
        <option value="">-- Select --</option>
        <option value="protected" <?= (isset($edit_tpl['use_case_type']) && $edit_tpl['use_case_type'] === 'protected') ? 'selected' : '' ?>>Protected (Verified Teacher)</option>
        <option value="general_teacher" <?= (isset($edit_tpl['use_case_type']) && $edit_tpl['use_case_type'] === 'general_teacher') ? 'selected' : '' ?>>General Teacher</option>
        <option value="student" <?= (isset($edit_tpl['use_case_type']) && $edit_tpl['use_case_type'] === 'student') ? 'selected' : '' ?>>Student Bulk</option>
    </select>
</label><br>
    <label>Fields (comma-separated, e.g. NAME,SCHOOL,DATE):
        <input type="text" name="fields" id="fieldsInput" required value="<?= isset($edit_tpl['field_definitions']) ? htmlspecialchars(implode(',', json_decode($edit_tpl['field_definitions'], true))) : '' ?>">
    </label><br>
    <label>PDF File:
        <input type="file" name="pdf" id="pdfInput" accept="application/pdf" <?= $edit_tpl ? '' : 'required' ?>>
       
    </label><br>
    <button type="button" id="previewBtn" class="btn">Preview</button>
    <input type="hidden" name="field_coordinates" id="fieldCoordinates" value="<?= htmlspecialchars($edit_tpl['field_coordinates'] ?? '') ?>">
    <br><br><div style="text-align:center;">
    <button type="submit" class="btn"><?= $edit_tpl ? 'Update Template' : 'Upload Template' ?></button>
    <?php if ($edit_tpl): ?>
        <a href="templates.php" class="btn" style="background:#bbb;">Cancel Edit</a>
    <?php endif; ?>
    </div>
</form>

    <!-- Modal is OUTSIDE the form -->
    <div id="pdfPreviewModal" style="display:none;position:fixed;top:0;left:0;width:100vw;height:100vh;z-index:1000;background:rgba(0,0,0,0.85);align-items:center;justify-content:center;">
  <div id="pdfPreviewInner" style="position:relative;width:100vw;height:100vh;background:#fff;padding:0;box-sizing:border-box;overflow:auto;max-width:none;max-height:none;margin:0;border-radius:0;display:flex;flex-direction:column;">
    <div style="display:flex;justify-content:center;align-items:center;padding:10px 20px 0 20px;position:relative;">
      <!-- Page Navigation Controls - Centered -->
      <div id="pageNavigation" style="display:flex;align-items:center;gap:15px;">
        <button type="button" id="prevPageBtn" style="font-size:16px;background:#007bff;color:#fff;border:none;border-radius:6px;padding:8px 20px;cursor:pointer;font-weight:600;box-shadow:0 2px 4px rgba(0,0,0,0.1);">◀ Prev</button>
        <span id="pageInfo" style="font-size:18px;font-weight:bold;color:#333;min-width:120px;text-align:center;background:#f8f9fa;padding:8px 20px;border-radius:6px;border:2px solid #dee2e6;">Page <span id="currentPageNum">1</span> of <span id="totalPages">1</span></span>
        <button type="button" id="nextPageBtn" style="font-size:16px;background:#007bff;color:#fff;border:none;border-radius:6px;padding:8px 20px;cursor:pointer;font-weight:600;box-shadow:0 2px 4px rgba(0,0,0,0.1);">Next ▶</button>
      </div>
      <!-- Action Buttons - Absolute positioned on right -->
      <div style="position:absolute;right:20px;display:flex;gap:12px;">
        <button type="button" id="fullScreenBtn" style="font-size:16px;background:#222;color:#fff;border:none;border-radius:4px;padding:6px 16px;cursor:pointer;">Full Screen</button>
        <button type="button" id="closePdfPreview" style="font-size:22px;background:#e74c3c;color:#fff;border:none;border-radius:4px;padding:6px 14px;cursor:pointer;">&times;</button>
      </div>
    </div>
    <div style="margin-bottom:6px;padding:0 20px;">
      <!-- Line 1: Field Controls with Action Buttons -->
      <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:4px;">
        <div>
          <label>Choose Placeholder:
            <select id="placeholderType"></select>
          </label>
          <label style="margin-left:20px;">Font Size:
            <select id="fontSizeSelect">
              <option value="14">14px</option>
              <option value="16">16px</option>
              <option value="18">18px</option>
              <option value="20">20px</option>
              <option value="24">24px</option>
              <option value="28">28px</option>
              <option value="32">32px</option>
            </select>
          </label>
          <label style="margin-left:20px;">Font Color:
            <input type="color" id="fontColorPicker" value="#000000" style="vertical-align:middle;width:32px;height:24px;border:none;">
          </label>
        </div>
        <div>
          <button type="button" id="fullCertificatePreviewBtn" style="padding:8px 16px;background:#28a745;color:white;border:none;border-radius:4px;cursor:pointer;font-size:12px;font-weight:bold;margin-right:8px;">
            🎓 Preview Full Certificate
          </button>
          <button type="button" id="downloadSampleBtn" style="padding:8px 16px;background:#007bff;color:white;border:none;border-radius:4px;cursor:pointer;font-size:12px;font-weight:bold;">
            📥 Download Sample PDF
          </button>
        </div>
      </div>
      
      <!-- Line 2: Alignment Controls with Status -->
      <div style="display:flex;justify-content:space-between;align-items:center;">
        <div>
          <label>Alignment:
            <select id="alignmentOverride" style="vertical-align:middle;">
              <option value="auto">🎯 Smart Auto</option>
              <option value="LEFT">⬅️ Left</option>
              <option value="CENTER">⬛ Center</option>
              <option value="RIGHT">➡️ Right</option>
            </select>
          </label>
          <label style="margin-left:20px;display:inline-flex;align-items:center;gap:6px;">
            Show Alignment Indicators
            <input type="checkbox" id="showAlignmentIndicators" checked style="vertical-align:middle;margin:0;">
          </label>
          <button type="button" id="clearSelectionBtn" style="margin-left:10px;padding:6px 12px;font-size:12px;background:#6c757d;color:white;border:none;border-radius:4px;cursor:pointer;font-weight:500;">Clear Selection</button>
        </div>
        <div>
          <span id="placementHint" style="color:#888;">Click on PDF to place marker</span>
          <span style="margin-left:20px;font-size:12px;color:#999;">Arrows: Move • Shift+Arrows: Fine • Del: Delete • Esc: Deselect • Ctrl+◀▶: Pages</span>
        </div>
      </div>
    </div>
    <div id="pdfPreviewContainer" style="flex:1;position:relative;width:100vw;height:100%;overflow:auto;background:#f8f8f8;display:flex;align-items:center;justify-content:center;"></div>
  </div>
</div>
    </div>



    <div class="user-table-section">
        <h3>All Templates</h3>
        <div class="table-responsive">
            <table>
                <tr>
                    <th>ID</th><th>Name</th><th>Use Case</th><th>Fields</th><th>Created By</th><th>Actions</th>
                </tr>
                <?php foreach ($templates as $tpl): ?>
                <tr>
                    <td><?= htmlspecialchars($tpl['id']) ?></td>
                    <td><?= htmlspecialchars($tpl['name']) ?></td>
                    <td><?= htmlspecialchars($tpl['use_case_type']) ?></td>
                    <td><?= htmlspecialchars(implode(', ', json_decode($tpl['field_definitions'], true))) ?></td>
                    <td><?= htmlspecialchars($tpl['created_by'] ?? '') ?></td>
                    <td style="text-align:center;">
    <span style="display:inline-flex;align-items:center;gap:6px;">
        <a href="templates.php?edit=<?= $tpl['id'] ?>">
            <img src="../img/edit.png" alt="Edit" title="Edit" style="width:18px;height:18px;vertical-align:middle;">
        </a>
        <a href="templates.php?validate=<?= $tpl['id'] ?>" title="Re-validate template">
            <img src="../img/download.png" alt="Validate" title="Re-validate" style="width:18px;height:18px;vertical-align:middle;">
        </a>
        <a href="templates.php?delete=<?= $tpl['id'] ?>" onclick="return confirm('Delete this template?')">
            <img src="../img/delete.png" alt="Delete" title="Delete" style="width:18px;height:18px;vertical-align:middle;">
        </a>
    </span>
</td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
  

    <script>
// Template management functions

let pdfDoc = null, scale = 1.2, markers = [], currentPage = 1, totalPages = 1;
const markerColors = { NAME: '#007bff', DATE: '#28a745', LOGO: '#ffc107' };

// Dynamically populate placeholderType dropdown from fields input
function updatePlaceholderDropdown() {
    const fields = document.getElementById('fieldsInput').value.split(',').map(f => f.trim()).filter(Boolean);
    const select = document.getElementById('placeholderType');
    select.innerHTML = '';
    fields.forEach(f => {
        const opt = document.createElement('option');
        opt.value = f;
        opt.textContent = `{${f}}`;
        select.appendChild(opt);
    });
    // Optionally add static placeholders like LOGO
    const logoOpt = document.createElement('option');
    logoOpt.value = 'LOGO';
    logoOpt.textContent = '{LOGO}';
    select.appendChild(logoOpt);
}

// Update preview when placeholder type changes
document.getElementById('placeholderType').addEventListener('change', function() {
    if (selectedMarker) {
        selectedMarker.type = this.value;
        updatePreviewText();
        
        // Re-render markers
        if (pdfDoc) {
            const canvas = document.getElementById('pdfCanvas');
            if (canvas) {
                const width = parseFloat(canvas.style.width);
                const height = parseFloat(canvas.style.height);
                renderMarkers(currentPage, width, height);
            }
        }
        
        updateMarkerData();
    }
});
document.getElementById('fieldsInput').addEventListener('input', updatePlaceholderDropdown);
window.onload = updatePlaceholderDropdown;

document.getElementById('previewBtn').onclick = async function() {
    const file = document.getElementById('pdfInput').files[0];
    if (!file) return alert('Please select a PDF file.');
    // Show loader
    document.getElementById('globalLoaderOverlay').classList.remove('hidden');
    // Show modal
    document.getElementById('pdfPreviewModal').style.display = 'flex';
    // Clear previous preview
    document.getElementById('pdfPreviewContainer').innerHTML = '';
    const reader = new FileReader();
    reader.onload = async function(e) {
        const typedarray = new Uint8Array(e.target.result);
        pdfDoc = await pdfjsLib.getDocument({data: typedarray}).promise;
        totalPages = pdfDoc.numPages;
        currentPage = 1;
        
        // Update page info display
        document.getElementById('totalPages').textContent = totalPages;
        document.getElementById('currentPageNum').textContent = currentPage;
        
        // Show/hide page navigation based on page count
        updatePageNavigation();
        
        renderPage(currentPage);
        // Hide loader after rendering
        document.getElementById('globalLoaderOverlay').classList.add('hidden');
    };
    reader.readAsArrayBuffer(file);
};

// Hide loader after renderPage completes (also in renderPage)
function renderPage(num) {
    pdfDoc.getPage(num).then(function(page) {
        // ... (existing code)
        page.render({
            canvasContext: ctx,
            viewport: page.getViewport({scale: ptToPx * dpr})
        }).promise.then(() => {
            // ... (existing code)
            // Hide loader after rendering
            document.getElementById('globalLoaderOverlay').classList.add('hidden');
        });
        // ... (existing code)
    });
}

// Show loader on form submit (template upload)
document.querySelector('form').onsubmit = function() {
    document.getElementById('globalLoaderOverlay').classList.remove('hidden');
    return true;
};

// Only the Full Screen button should trigger fullscreen
// (no code here for previewBtn)
document.getElementById('fullScreenBtn').onclick = function() {
    var modal = document.getElementById('pdfPreviewModal');
    if (!document.fullscreenElement) {
        if (modal.requestFullscreen) {
            modal.requestFullscreen();
        } else if (modal.webkitRequestFullscreen) { /* Safari */
            modal.webkitRequestFullscreen();
        } else if (modal.msRequestFullscreen) { /* IE11 */
            modal.msRequestFullscreen();
        }
        this.textContent = 'Exit Full Screen';
    } else {
        if (document.exitFullscreen) {
            document.exitFullscreen();
        } else if (document.webkitExitFullscreen) { /* Safari */
            document.webkitExitFullscreen();
        } else if (document.msExitFullscreen) { /* IE11 */
            document.msExitFullscreen();
        }
        this.textContent = 'Full Screen';
    }
};
document.addEventListener('fullscreenchange', function() {
    var btn = document.getElementById('fullScreenBtn');
    if (!document.fullscreenElement) {
        btn.textContent = 'Full Screen';
    } else {
        btn.textContent = 'Exit Full Screen';
    }
});

document.getElementById('closePdfPreview').onclick = function(e) {
    e.preventDefault();
    // Exit fullscreen if active
    if (document.fullscreenElement) {
        if (document.exitFullscreen) {
            document.exitFullscreen();
        } else if (document.webkitExitFullscreen) {
            document.webkitExitFullscreen();
        } else if (document.msExitFullscreen) {
            document.msExitFullscreen();
        }
        // Wait for fullscreenchange event to hide modal
        document.addEventListener('fullscreenchange', function handler() {
            if (!document.fullscreenElement) {
                document.getElementById('pdfPreviewModal').style.display = 'none';
                document.getElementById('pdfPreviewContainer').innerHTML = '';
                document.removeEventListener('fullscreenchange', handler);
            }
        });
    } else {
        document.getElementById('pdfPreviewModal').style.display = 'none';
        document.getElementById('pdfPreviewContainer').innerHTML = '';
    }
};
function renderPage(num) {
    pdfDoc.getPage(num).then(function(page) {
        // Get the actual PDF dimensions in points (1pt = 1/72 inch)
        const viewport = page.getViewport({scale: 1});
        // Calculate the screen DPI (dots per inch)
        // 1 CSS inch = 96 CSS px, but actual device DPI may differ
        let dpi = 96; // fallback
        if (window.screen && window.screen.deviceXDPI) {
            dpi = window.screen.deviceXDPI;
        } else {
            // Use a temporary div to measure true DPI
            const div = document.createElement('div');
            div.style.width = '1in';
            div.style.height = '1in';
            div.style.position = 'absolute';
            div.style.left = '-1000px';
            document.body.appendChild(div);
            dpi = div.offsetWidth;
            document.body.removeChild(div);
        }
        const ptToPx = dpi / 72;
        const actualWidthPx = viewport.width * ptToPx;
        const actualHeightPx = viewport.height * ptToPx;
        // For sharpness on retina screens, use devicePixelRatio
        const dpr = window.devicePixelRatio || 1;

        // Render canvas at true PDF pixel size
        const canvas = document.createElement('canvas');
        canvas.width = Math.round(actualWidthPx * dpr);
        canvas.height = Math.round(actualHeightPx * dpr);
        canvas.style.width = actualWidthPx + 'px';
        canvas.style.height = actualHeightPx + 'px';
        canvas.style.border = '1px solid #aaa';
        canvas.style.display = 'block';
        canvas.style.position = 'relative';
        canvas.id = 'pdfCanvas';

        // Marker overlay
        const markerLayer = document.createElement('div');
        markerLayer.style.position = 'absolute';
        markerLayer.style.left = 0;
        markerLayer.style.top = 0;
        markerLayer.style.width = actualWidthPx + 'px';
        markerLayer.style.height = actualHeightPx + 'px';
        markerLayer.style.pointerEvents = 'none';
        markerLayer.id = 'markerLayer';

        // Container for centering
        const container = document.createElement('div');
        container.style.position = 'relative';
        container.style.width = actualWidthPx + 'px';
        container.style.height = actualHeightPx + 'px';
        container.style.display = 'flex';
        container.style.alignItems = 'center';
        container.style.justifyContent = 'center';
        container.appendChild(canvas);
        container.appendChild(markerLayer);

        const previewDiv = document.getElementById('pdfPreviewContainer');
        previewDiv.innerHTML = '';
        previewDiv.style.display = 'flex';
        previewDiv.style.alignItems = 'center';
        previewDiv.style.justifyContent = 'center';
        previewDiv.appendChild(container);

        // Render PDF page
        const ctx = canvas.getContext('2d');
        ctx.setTransform(dpr, 0, 0, dpr, 0, 0); // for retina sharpness
        page.render({
            canvasContext: ctx,
            viewport: page.getViewport({scale: ptToPx * dpr})
        }).promise.then(() => {
            markerLayer.style.pointerEvents = 'auto';
            markerLayer.onclick = function(e) {
                // Check if click was on empty space (not on a marker)
                if (e.target === markerLayer) {
                    const rect = markerLayer.getBoundingClientRect();
                    const x = e.clientX - rect.left;
                    const y = e.clientY - rect.top;
                    const type = document.getElementById('placeholderType').value;
                    
                    // Clear selection when placing new marker
                    clearMarkerSelection();
                    addMarker(type, num, x, y);
                }
            };
            renderMarkers(num, actualWidthPx, actualHeightPx);
        });

        // Debug overlay for DPI
        const dpiDebug = document.createElement('div');
        dpiDebug.style.position = 'absolute';
        dpiDebug.style.bottom = '8px';
        dpiDebug.style.right = '8px';
        dpiDebug.style.background = 'rgba(0,0,0,0.7)';
        dpiDebug.style.color = '#fff';
        dpiDebug.style.fontSize = '13px';
        dpiDebug.style.padding = '3px 8px';
        dpiDebug.style.borderRadius = '6px';
        dpiDebug.style.zIndex = 100;
        dpiDebug.textContent = `Page ${num}/${totalPages} • DPI: ${Math.round(dpi)}, dpr: ${dpr}, px: ${Math.round(actualWidthPx)}x${Math.round(actualHeightPx)}`;
        container.appendChild(dpiDebug);
    });
}

function addMarker(type, page, x, y) {
    // Use the actual PDF canvas size for mapping
    const pdfCanvas = document.getElementById('pdfCanvas');
    let previewWidth = 0, previewHeight = 0;
    if (pdfCanvas) {
        // The canvas is rendered at device pixels, but style.width is CSS pixels
        previewWidth = parseFloat(pdfCanvas.style.width);
        previewHeight = parseFloat(pdfCanvas.style.height);
    } else {
        // Fallback to container size if canvas not found
        const previewDiv = document.getElementById('pdfPreviewContainer');
        previewWidth = previewDiv.scrollWidth;
        previewHeight = previewDiv.scrollHeight;
    }
    const fontSize = document.getElementById('fontSizeSelect').value;
    const fontColor = document.getElementById('fontColorPicker').value;
    const alignmentOverride = document.getElementById('alignmentOverride').value;

    // Check if a marker of this type already exists on this page
    let existing = markers.find(m => m.type === type && m.page === page);
    if (existing) {
        // Move the existing marker and update position
        existing.x = x;
        existing.y = y;
        existing.preview_width = previewWidth;
        existing.preview_height = previewHeight;
        
        // Only update other properties if this marker is selected or no marker is selected
        if (!selectedMarker || selectedMarker.id === existing.id) {
            existing.font_size = fontSize;
            existing.font_color = fontColor;
            existing.alignment_override = alignmentOverride;
        }
        
        // Select this marker for editing
        selectMarkerForEditing(existing);
    } else {
        // Add a new marker
        const marker = {
            type,
            page,
            x,
            y,
            font_size: fontSize,
            font_color: fontColor,
            alignment_override: alignmentOverride,
            preview_width: previewWidth,
            preview_height: previewHeight,
            id: Date.now()
        };
        markers.push(marker);
        
        // Select the new marker for editing
        selectMarkerForEditing(marker);
    }
    renderMarkers(page, previewWidth, previewHeight);
    updateMarkerData();
}

function renderMarkers(page, width, height) {
    const markerLayer = document.getElementById('markerLayer');
    if (!markerLayer) return;
    markerLayer.innerHTML = '';
    
    // Add zone indicators first (behind markers) if enabled
    const showIndicators = document.getElementById('showAlignmentIndicators').checked;
    if (showIndicators) {
        addZoneIndicators(markerLayer, width, height);
    }
    
    markers.filter(m => m.page === page).forEach(marker => {
        // Calculate intelligent alignment for this marker
        const alignmentInfo = calculatePreviewAlignment(marker, width, height);
        
        // Create marker element
        const el = document.createElement('div');
        const markerText = alignmentInfo.alignmentSource === 'manual' ? 
            `{${marker.type}} 🎯` : `{${marker.type}}`;
        el.innerText = markerText;
        el.style.position = 'absolute';
        el.style.left = (marker.x - 30) + 'px';
        el.style.top = (marker.y - 12) + 'px';
        el.style.background = markerColors[marker.type] || '#aaa';
        el.style.color = marker.font_color || '#fff';
        el.style.padding = '2px 8px';
        el.style.borderRadius = '4px';
        el.style.cursor = 'pointer';
        el.style.fontWeight = 'bold';
        el.style.fontSize = (marker.font_size || 16) + 'px';
        el.draggable = true;
        
        // Add special styling for manual alignment
        if (alignmentInfo.alignmentSource === 'manual') {
            el.style.border = '2px solid #9b59b6';
            el.style.boxShadow = '0 0 6px rgba(155, 89, 182, 0.4)';
        }
        
        // Highlight selected marker
        if (selectedMarker && selectedMarker.id === marker.id) {
            el.style.border = '3px solid #f39c12';
            el.style.boxShadow = '0 0 10px rgba(243, 156, 18, 0.6)';
            el.style.transform = 'scale(1.1)';
            el.style.zIndex = '1000';
        }
        const alignmentDisplay = alignmentInfo.alignmentSource === 'manual' ? 
            `${alignmentInfo.alignment} (Manual Override)` : 
            `${alignmentInfo.alignment} (Smart Auto)`;
        el.title = `Field: {${marker.type}}\nAlignment: ${alignmentDisplay}\nZone: ${alignmentInfo.zone} (${alignmentInfo.xPercent}%, ${alignmentInfo.yPercent}%)\n\nDrag to move • Double-click to delete`;

        // Add alignment indicator if enabled
        if (showIndicators) {
            const alignmentIndicator = createAlignmentIndicator(marker, alignmentInfo, width, height);
            markerLayer.appendChild(alignmentIndicator);
        }
        
        // Drag logic
        el.ondragstart = function(e) {
            e.dataTransfer.setData('text/plain', marker.id);
            window._dragMarkerId = marker.id;
        };
        el.ondragend = function(e) {
            const rect = markerLayer.getBoundingClientRect();
            const x = e.clientX - rect.left;
            const y = e.clientY - rect.top;
            marker.x = x;
            marker.y = y;
            renderMarkers(page, width, height);
            updateMarkerData();
        };
        // Remove on double-click
        el.ondblclick = function() {
            markers = markers.filter(m => m.id !== marker.id);
            renderMarkers(page, width, height);
            updateMarkerData();
        };
        
        // Single click to select and edit this marker
        el.onclick = function(e) {
            e.stopPropagation();
            selectMarkerForEditing(marker);
        };
        
        markerLayer.appendChild(el);
    });
}

function updateMarkerData() {
    document.getElementById('fieldCoordinates').value = JSON.stringify(markers);
}

let selectedMarker = null;

// Sample text for different field types
const sampleTexts = {
    'NAME': 'John Smith',
    'STUDENT_NAME': 'Sarah Johnson',
    'PARTICIPANT_NAME': 'Michael Brown',
    'COURSE': 'Advanced Web Development',
    'COURSE_NAME': 'Digital Marketing Fundamentals',
    'TITLE': 'Certificate of Completion',
    'DATE': 'December 15, 2024',
    'COMPLETION_DATE': 'November 30, 2024',
    'ISSUE_DATE': 'January 10, 2025',
    'SCHOOL': 'Tech Academy Institute',
    'INSTITUTION': 'University of Excellence',
    'ORGANIZATION': 'Professional Learning Center',
    'INSTRUCTOR': 'Dr. Emily Davis',
    'TEACHER': 'Prof. Robert Wilson',
    'DIRECTOR': 'Jane Anderson',
    'SIGNATURE': 'Director Signature',
    'GRADE': 'A+',
    'SCORE': '95%',
    'HOURS': '40 Hours',
    'CREDITS': '3 Credits',
    'ID': 'CERT-2024-001',
    'CERTIFICATE_ID': 'CRT-ABC-123',
    'LOGO': '[LOGO]'
};

function getSampleText(fieldType) {
    const upperType = fieldType.toUpperCase();
    
    // Direct match
    if (sampleTexts[upperType]) {
        return sampleTexts[upperType];
    }
    
    // Partial matches
    if (upperType.includes('NAME')) return sampleTexts['NAME'];
    if (upperType.includes('COURSE')) return sampleTexts['COURSE'];
    if (upperType.includes('DATE')) return sampleTexts['DATE'];
    if (upperType.includes('SCHOOL') || upperType.includes('INSTITUTION')) return sampleTexts['SCHOOL'];
    if (upperType.includes('INSTRUCTOR') || upperType.includes('TEACHER')) return sampleTexts['INSTRUCTOR'];
    if (upperType.includes('SIGNATURE')) return sampleTexts['SIGNATURE'];
    if (upperType.includes('GRADE') || upperType.includes('SCORE')) return sampleTexts['GRADE'];
    if (upperType.includes('ID')) return sampleTexts['ID'];
    if (upperType.includes('LOGO')) return sampleTexts['LOGO'];
    
    // Default
    return `Sample ${fieldType}`;
}

function selectMarkerForEditing(marker) {
    selectedMarker = marker;
    
    // Update UI controls to show this marker's settings
    document.getElementById('placeholderType').value = marker.type;
    document.getElementById('fontSizeSelect').value = marker.font_size || 16;
    document.getElementById('fontColorPicker').value = marker.font_color || '#000000';
    document.getElementById('alignmentOverride').value = marker.alignment_override || 'auto';
    
    // Update the hint text to show which marker is selected
    const hint = document.getElementById('placementHint');
    hint.innerHTML = `<strong>Editing: {${marker.type}}</strong>`;
    hint.style.color = '#f39c12';
    hint.style.fontWeight = 'bold';
    
    // Highlight the selected marker
    renderMarkers(marker.page, 
        parseFloat(document.getElementById('pdfCanvas').style.width),
        parseFloat(document.getElementById('pdfCanvas').style.height)
    );
}

function clearMarkerSelection() {
    selectedMarker = null;
    const hint = document.getElementById('placementHint');
    hint.textContent = 'Click on PDF to place marker';
    hint.style.color = '#888';
    hint.style.fontWeight = 'normal';
    
}



// Calculate intelligent alignment preview (mirrors server-side logic)
function calculatePreviewAlignment(marker, pdfWidth, pdfHeight) {
    const x = marker.x;
    const y = marker.y;
    const fieldType = marker.type.toLowerCase();
    
    // Calculate percentages
    const xPercent = (x / pdfWidth) * 100;
    const yPercent = (y / pdfHeight) * 100;
    
    // Zone detection
    const isInMainZone = yPercent >= 25 && yPercent <= 75;
    const isInCenterHorizontal = xPercent >= 20 && xPercent <= 80;
    
    let alignment = 'LEFT';
    let zone = 'Header';
    let cellWidth = 150;
    let alignmentSource = 'smart';
    
    if (yPercent >= 25 && yPercent <= 75) {
        zone = 'Main';
    } else if (yPercent > 75) {
        zone = 'Footer';
    }
    
    // Check for user alignment override first
    if (marker.alignment_override && marker.alignment_override !== 'auto') {
        alignment = marker.alignment_override;
        alignmentSource = 'manual';
        
        // Adjust cell width based on manual alignment
        switch (alignment) {
            case 'CENTER':
                cellWidth = fieldType.includes('name') ? 300 : 
                           fieldType.includes('course') || fieldType.includes('title') ? 350 : 200;
                break;
            case 'LEFT':
                cellWidth = 200;
                break;
            case 'RIGHT':
                cellWidth = 150;
                break;
        }
    } else {
        // Apply intelligent alignment rules (auto mode)
        if (fieldType.includes('name')) {
            if (isInMainZone) {
                alignment = 'CENTER';
                cellWidth = 300;
            } else {
                alignment = 'CENTER';
                cellWidth = 200;
            }
        } else if (fieldType.includes('course') || fieldType.includes('title') || fieldType.includes('subject')) {
            if (isInMainZone) {
                alignment = 'CENTER';
                cellWidth = 350;
            } else {
                alignment = 'LEFT';
                cellWidth = 200;
            }
        } else if (fieldType.includes('date')) {
            alignment = xPercent < 50 ? 'LEFT' : 'RIGHT';
            cellWidth = 150;
        } else if (fieldType.includes('signature') || fieldType.includes('instructor') || fieldType.includes('director')) {
            alignment = 'CENTER';
            cellWidth = 180;
        } else if (fieldType.includes('school') || fieldType.includes('organization') || fieldType.includes('institution')) {
            if (isInMainZone && isInCenterHorizontal) {
                alignment = 'CENTER';
                cellWidth = 250;
            } else {
                alignment = 'LEFT';
                cellWidth = 200;
            }
        } else {
            // Default field logic
            if (isInCenterHorizontal && isInMainZone) {
                alignment = 'CENTER';
                cellWidth = 200;
            } else if (xPercent >= 80) {
                alignment = 'RIGHT';
                cellWidth = 150;
            } else {
                alignment = 'LEFT';
                cellWidth = 150;
            }
        }
    }
    
    return {
        alignment: alignment,
        zone: zone,
        cellWidth: cellWidth,
        alignmentSource: alignmentSource,
        xPercent: Math.round(xPercent),
        yPercent: Math.round(yPercent)
    };
}

// Create visual alignment indicator
function createAlignmentIndicator(marker, alignmentInfo, pdfWidth, pdfHeight) {
    const indicator = document.createElement('div');
    indicator.style.position = 'absolute';
    indicator.style.pointerEvents = 'none';
    indicator.style.zIndex = '1';
    
    // Calculate indicator position and size
    let indicatorX = marker.x;
    let indicatorWidth = alignmentInfo.cellWidth;
    
    if (alignmentInfo.alignment === 'CENTER') {
        indicatorX = marker.x - (indicatorWidth / 2);
    } else if (alignmentInfo.alignment === 'RIGHT') {
        indicatorX = marker.x - indicatorWidth;
    }
    
    // Ensure indicator stays within bounds
    indicatorX = Math.max(0, Math.min(indicatorX, pdfWidth - indicatorWidth));
    
    indicator.style.left = indicatorX + 'px';
    indicator.style.top = (marker.y - 2) + 'px';
    indicator.style.width = indicatorWidth + 'px';
    indicator.style.height = '20px';
    
    // Style based on alignment type and source
    let alignmentColors;
    if (alignmentInfo.alignmentSource === 'manual') {
        // Purple tones for manual overrides
        alignmentColors = {
            'LEFT': 'rgba(155, 89, 182, 0.2)',    // Purple
            'CENTER': 'rgba(155, 89, 182, 0.2)',  // Purple
            'RIGHT': 'rgba(155, 89, 182, 0.2)'    // Purple
        };
    } else {
        // Original colors for smart auto alignment
        alignmentColors = {
            'LEFT': 'rgba(52, 152, 219, 0.2)',    // Blue
            'CENTER': 'rgba(46, 204, 113, 0.2)',  // Green  
            'RIGHT': 'rgba(231, 76, 60, 0.2)'     // Red
        };
    }
    
    indicator.style.background = alignmentColors[alignmentInfo.alignment];
    indicator.style.border = `2px dashed ${alignmentColors[alignmentInfo.alignment].replace('0.2', '0.6')}`;
    indicator.style.borderRadius = '4px';
    
    // Add alignment text
    const alignmentText = document.createElement('div');
    alignmentText.style.position = 'absolute';
    alignmentText.style.top = '-18px';
    alignmentText.style.left = '50%';
    alignmentText.style.transform = 'translateX(-50%)';
    alignmentText.style.fontSize = '10px';
    alignmentText.style.fontWeight = 'bold';
    alignmentText.style.color = alignmentColors[alignmentInfo.alignment].replace('0.2', '0.8');
    alignmentText.style.background = 'rgba(255, 255, 255, 0.9)';
    alignmentText.style.padding = '1px 4px';
    alignmentText.style.borderRadius = '2px';
    alignmentText.style.whiteSpace = 'nowrap';
    const alignmentDisplay = alignmentInfo.alignmentSource === 'manual' ? 
        `${alignmentInfo.alignment} (Manual)` : 
        `${alignmentInfo.alignment} (Auto)`;
    alignmentText.textContent = `${alignmentDisplay} (${alignmentInfo.zone})`;
    
    indicator.appendChild(alignmentText);
    
    return indicator;
}

// Add zone indicators to show certificate layout zones
function addZoneIndicators(markerLayer, width, height) {
    // Header zone (top 25%)
    const headerZone = document.createElement('div');
    headerZone.style.position = 'absolute';
    headerZone.style.left = '0px';
    headerZone.style.top = '0px';
    headerZone.style.width = width + 'px';
    headerZone.style.height = (height * 0.25) + 'px';
    headerZone.style.background = 'rgba(155, 89, 182, 0.1)';
    headerZone.style.border = '1px dashed rgba(155, 89, 182, 0.3)';
    headerZone.style.pointerEvents = 'none';
    headerZone.style.zIndex = '0';
    
    const headerLabel = document.createElement('div');
    headerLabel.style.position = 'absolute';
    headerLabel.style.top = '5px';
    headerLabel.style.left = '10px';
    headerLabel.style.fontSize = '12px';
    headerLabel.style.fontWeight = 'bold';
    headerLabel.style.color = 'rgba(155, 89, 182, 0.8)';
    headerLabel.style.background = 'rgba(255, 255, 255, 0.9)';
    headerLabel.style.padding = '2px 6px';
    headerLabel.style.borderRadius = '3px';
    headerLabel.textContent = 'Header Zone';
    headerZone.appendChild(headerLabel);
    
    // Main zone (middle 50%)
    const mainZone = document.createElement('div');
    mainZone.style.position = 'absolute';
    mainZone.style.left = '0px';
    mainZone.style.top = (height * 0.25) + 'px';
    mainZone.style.width = width + 'px';
    mainZone.style.height = (height * 0.5) + 'px';
    mainZone.style.background = 'rgba(46, 204, 113, 0.1)';
    mainZone.style.border = '1px dashed rgba(46, 204, 113, 0.3)';
    mainZone.style.pointerEvents = 'none';
    mainZone.style.zIndex = '0';
    
    const mainLabel = document.createElement('div');
    mainLabel.style.position = 'absolute';
    mainLabel.style.top = '5px';
    mainLabel.style.left = '10px';
    mainLabel.style.fontSize = '12px';
    mainLabel.style.fontWeight = 'bold';
    mainLabel.style.color = 'rgba(46, 204, 113, 0.8)';
    mainLabel.style.background = 'rgba(255, 255, 255, 0.9)';
    mainLabel.style.padding = '2px 6px';
    mainLabel.style.borderRadius = '3px';
    mainLabel.textContent = 'Main Zone (Smart Centering)';
    mainZone.appendChild(mainLabel);
    
    // Footer zone (bottom 25%)
    const footerZone = document.createElement('div');
    footerZone.style.position = 'absolute';
    footerZone.style.left = '0px';
    footerZone.style.top = (height * 0.75) + 'px';
    footerZone.style.width = width + 'px';
    footerZone.style.height = (height * 0.25) + 'px';
    footerZone.style.background = 'rgba(230, 126, 34, 0.1)';
    footerZone.style.border = '1px dashed rgba(230, 126, 34, 0.3)';
    footerZone.style.pointerEvents = 'none';
    footerZone.style.zIndex = '0';
    
    const footerLabel = document.createElement('div');
    footerLabel.style.position = 'absolute';
    footerLabel.style.top = '5px';
    footerLabel.style.left = '10px';
    footerLabel.style.fontSize = '12px';
    footerLabel.style.fontWeight = 'bold';
    footerLabel.style.color = 'rgba(230, 126, 34, 0.8)';
    footerLabel.style.background = 'rgba(255, 255, 255, 0.9)';
    footerLabel.style.padding = '2px 6px';
    footerLabel.style.borderRadius = '3px';
    footerLabel.textContent = 'Footer Zone';
    footerZone.appendChild(footerLabel);
    
    markerLayer.appendChild(headerZone);
    markerLayer.appendChild(mainZone);
    markerLayer.appendChild(footerZone);
}

// Toggle alignment indicators
document.getElementById('showAlignmentIndicators').addEventListener('change', function() {
    if (pdfDoc) {
        const canvas = document.getElementById('pdfCanvas');
        if (canvas) {
            const width = parseFloat(canvas.style.width);
            const height = parseFloat(canvas.style.height);
            renderMarkers(currentPage, width, height);
        }
    }
});

// Update alignment when override changes (only for selected marker)
document.getElementById('alignmentOverride').addEventListener('change', function() {
    if (selectedMarker) {
        selectedMarker.alignment_override = this.value;
        
        // Re-render markers to show updated alignment
        if (pdfDoc) {
            const canvas = document.getElementById('pdfCanvas');
            if (canvas) {
                const width = parseFloat(canvas.style.width);
                const height = parseFloat(canvas.style.height);
                renderMarkers(currentPage, width, height);
            }
        }
        
        updateMarkerData();
    }
});

// Update font size when changed (only for selected marker)
document.getElementById('fontSizeSelect').addEventListener('change', function() {
    if (selectedMarker) {
        selectedMarker.font_size = this.value;
        
        // Re-render markers to show updated font size
        if (pdfDoc) {
            const canvas = document.getElementById('pdfCanvas');
            if (canvas) {
                const width = parseFloat(canvas.style.width);
                const height = parseFloat(canvas.style.height);
                renderMarkers(currentPage, width, height);
            }
        }
        
        updateMarkerData();
    }
});

// Update font color when changed (only for selected marker)
document.getElementById('fontColorPicker').addEventListener('change', function() {
    if (selectedMarker) {
        selectedMarker.font_color = this.value;
        
        // Re-render markers to show updated font color
        if (pdfDoc) {
            const canvas = document.getElementById('pdfCanvas');
            if (canvas) {
                const width = parseFloat(canvas.style.width);
                const height = parseFloat(canvas.style.height);
                renderMarkers(currentPage, width, height);
            }
        }
        
        updateMarkerData();
    }
});

// Clear selection button
document.getElementById('clearSelectionBtn').addEventListener('click', function() {
    clearMarkerSelection();
    
    // Re-render to remove selection highlighting
    if (pdfDoc) {
        const canvas = document.getElementById('pdfCanvas');
        if (canvas) {
            const width = parseFloat(canvas.style.width);
            const height = parseFloat(canvas.style.height);
            renderMarkers(currentPage, width, height);
        }
    }
});

// Multi-Page Navigation Functions
function updatePageNavigation() {
    const prevBtn = document.getElementById('prevPageBtn');
    const nextBtn = document.getElementById('nextPageBtn');
    const pageNav = document.getElementById('pageNavigation');
    
    // Show/hide navigation if single page
    if (totalPages <= 1) {
        pageNav.style.visibility = 'hidden';
    } else {
        pageNav.style.visibility = 'visible';
    }
    
    // Enable/disable buttons based on current page
    prevBtn.disabled = (currentPage <= 1);
    nextBtn.disabled = (currentPage >= totalPages);
    
    // Update button styles
    prevBtn.style.opacity = prevBtn.disabled ? '0.5' : '1';
    prevBtn.style.cursor = prevBtn.disabled ? 'not-allowed' : 'pointer';
    nextBtn.style.opacity = nextBtn.disabled ? '0.5' : '1';
    nextBtn.style.cursor = nextBtn.disabled ? 'not-allowed' : 'pointer';
}

// Previous Page Button
document.getElementById('prevPageBtn').onclick = function() {
    if (currentPage > 1) {
        currentPage--;
        document.getElementById('currentPageNum').textContent = currentPage;
        renderPage(currentPage);
        updatePageNavigation();
    }
};

// Next Page Button
document.getElementById('nextPageBtn').onclick = function() {
    if (currentPage < totalPages) {
        currentPage++;
        document.getElementById('currentPageNum').textContent = currentPage;
        renderPage(currentPage);
        updatePageNavigation();
    }
};

// Keyboard shortcuts for page navigation and marker movement
document.addEventListener('keydown', function(e) {
    // Only work when modal is open
    const modal = document.getElementById('pdfPreviewModal');
    if (modal.style.display !== 'flex') return;
    
    // Arrow keys to move selected marker (with fine control using Shift)
    if (selectedMarker && !e.ctrlKey && (e.key === 'ArrowUp' || e.key === 'ArrowDown' || e.key === 'ArrowLeft' || e.key === 'ArrowRight')) {
        e.preventDefault();
        
        // Movement step: 1px for fine control (Shift), 10px for normal
        const step = e.shiftKey ? 1 : 10;
        
        switch(e.key) {
            case 'ArrowUp':
                selectedMarker.y -= step;
                break;
            case 'ArrowDown':
                selectedMarker.y += step;
                break;
            case 'ArrowLeft':
                selectedMarker.x -= step;
                break;
            case 'ArrowRight':
                selectedMarker.x += step;
                break;
        }
        
        // Re-render markers with updated position
        const canvas = document.getElementById('pdfCanvas');
        if (canvas) {
            const width = parseFloat(canvas.style.width);
            const height = parseFloat(canvas.style.height);
            renderMarkers(currentPage, width, height);
            updateMarkerData();
        }
    }
    // Page Up/Down for page navigation (only when no marker selected)
    else if (!selectedMarker && e.key === 'PageUp') {
        e.preventDefault();
        document.getElementById('prevPageBtn').click();
    } else if (!selectedMarker && e.key === 'PageDown') {
        e.preventDefault();
        document.getElementById('nextPageBtn').click();
    }
    // Ctrl+Arrow keys for page navigation (works even with marker selected)
    else if (e.ctrlKey && e.key === 'ArrowLeft') {
        e.preventDefault();
        document.getElementById('prevPageBtn').click();
    } else if (e.ctrlKey && e.key === 'ArrowRight') {
        e.preventDefault();
        document.getElementById('nextPageBtn').click();
    }
    // Ctrl+P for preview
    else if (e.ctrlKey && e.key === 'p') {
        e.preventDefault();
        document.getElementById('fullCertificatePreviewBtn').click();
    }
    // Ctrl+D for download
    else if (e.ctrlKey && e.key === 'd') {
        e.preventDefault();
        document.getElementById('downloadSampleBtn').click();
    }
    // Delete or Backspace to remove selected marker
    else if (selectedMarker && (e.key === 'Delete' || e.key === 'Backspace')) {
        e.preventDefault();
        markers = markers.filter(m => m.id !== selectedMarker.id);
        clearMarkerSelection();
        const canvas = document.getElementById('pdfCanvas');
        if (canvas) {
            const width = parseFloat(canvas.style.width);
            const height = parseFloat(canvas.style.height);
            renderMarkers(currentPage, width, height);
            updateMarkerData();
        }
    }
    // Escape to clear selection
    else if (e.key === 'Escape') {
        if (selectedMarker) {
            clearMarkerSelection();
            const canvas = document.getElementById('pdfCanvas');
            if (canvas) {
                const width = parseFloat(canvas.style.width);
                const height = parseFloat(canvas.style.height);
                renderMarkers(currentPage, width, height);
            }
        }
    }
});

// Optional: re-fit PDF on window resize
window.addEventListener('resize', function() {
    if (pdfDoc) renderPage(currentPage);
});

// Full Certificate Preview functionality
document.getElementById('fullCertificatePreviewBtn').addEventListener('click', function() {
    if (!pdfDoc || markers.length === 0) {
        alert('Please place some field markers first to see the preview.');
        return;
    }
    
    toggleFullCertificatePreview();
});

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    if (pdfDoc && markers.length > 0) {
        // Ctrl+P or Cmd+P for preview
        if ((e.ctrlKey || e.metaKey) && e.key === 'p') {
            e.preventDefault(); // Prevent browser print dialog
            toggleFullCertificatePreview();
        }
        // Ctrl+D or Cmd+D for download
        else if ((e.ctrlKey || e.metaKey) && e.key === 'd') {
            e.preventDefault(); // Prevent browser bookmark dialog
            downloadSamplePDF();
        }
    }
});

// Download Sample PDF functionality
document.getElementById('downloadSampleBtn').addEventListener('click', function() {
    if (!pdfDoc || markers.length === 0) {
        alert('Please place some field markers first to generate a sample PDF.');
        return;
    }
    
    downloadSamplePDF();
});

function downloadSamplePDF() {
    const button = document.getElementById('downloadSampleBtn');
    const originalText = button.innerHTML;
    
    // Show loading state
    button.innerHTML = '⏳ Generating PDF...';
    button.disabled = true;
    button.style.background = '#6c757d';
    
    // Prepare sample data for all fields
    const sampleFieldData = {};
    markers.forEach(marker => {
        sampleFieldData[marker.type] = getSampleText(marker.type);
    });
    
    // Create form data to send to server
    const formData = new FormData();
    formData.append('action', 'download_sample');
    formData.append('field_coordinates', JSON.stringify(markers));
    formData.append('field_values', JSON.stringify(sampleFieldData));
    
    // Get current template info if editing
    const editId = new URLSearchParams(window.location.search).get('edit');
    if (editId) {
        formData.append('template_id', editId);
    } else {
        // For new templates, we need the PDF file
        const pdfInput = document.getElementById('pdfInput');
        if (pdfInput.files[0]) {
            formData.append('pdf_file', pdfInput.files[0]);
        } else {
            alert('Please select a PDF file first.');
            resetDownloadButton();
            return;
        }
    }
    
    // Send request to server
    fetch('templates.php', {
        method: 'POST',
        body: formData
    })
    .then(response => {
        if (response.ok) {
            return response.blob();
        }
        throw new Error('Failed to generate PDF');
    })
    .then(blob => {
        // Create download link
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'sample-certificate.pdf';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);
        
        resetDownloadButton();
    })
    .catch(error => {
        console.error('Download error:', error);
        alert('Failed to generate PDF. Please try again.');
        resetDownloadButton();
    });
    
    function resetDownloadButton() {
        button.innerHTML = originalText;
        button.disabled = false;
        button.style.background = '#007bff';
    }
}

let isShowingFullPreview = false;

function toggleFullCertificatePreview() {
    const button = document.getElementById('fullCertificatePreviewBtn');
    const canvas = document.getElementById('pdfCanvas');
    const markerLayer = document.getElementById('markerLayer');
    
    if (!canvas || !markerLayer) return;
    
    if (isShowingFullPreview) {
        // Switch back to editing mode
        showEditingMode();
        button.innerHTML = '🎓 Preview Full Certificate';
        button.style.background = '#28a745';
        isShowingFullPreview = false;
    } else {
        // Switch to full preview mode
        showFullCertificatePreview();
        button.innerHTML = '✏️ Back to Editing';
        button.style.background = '#6c757d';
        isShowingFullPreview = true;
    }
}

function showFullCertificatePreview() {
    const canvas = document.getElementById('pdfCanvas');
    const markerLayer = document.getElementById('markerLayer');
    
    if (!canvas || !markerLayer) return;
    
    const width = parseFloat(canvas.style.width);
    const height = parseFloat(canvas.style.height);
    
    // Clear existing markers and indicators
    markerLayer.innerHTML = '';
    
    // Add all field texts with sample data - ONLY for current page
    markers.filter(m => m.page === currentPage).forEach(marker => {
        const sampleText = getSampleText(marker.type);
        const alignmentInfo = calculatePreviewAlignment(marker, width, height);
        
        // Create text element that looks like final output
        const textElement = document.createElement('div');
        textElement.style.position = 'absolute';
        textElement.style.fontFamily = 'Helvetica, Arial, sans-serif'; // Use Helvetica like PDF output
        textElement.style.fontSize = (marker.font_size || 16) + 'px';
        textElement.style.color = marker.font_color || '#000000';
        textElement.style.fontWeight = 'normal';
        textElement.style.lineHeight = '1.2';
        textElement.style.whiteSpace = 'nowrap';
        textElement.style.pointerEvents = 'none';
        textElement.textContent = sampleText;
        
        // Calculate position based on alignment
        let x = marker.x;
        let textAlign = 'left';
        let cellWidth = alignmentInfo.cellWidth;
        
        if (marker.alignment_override && marker.alignment_override !== 'auto') {
            // Use manual alignment
            switch (marker.alignment_override) {
                case 'CENTER':
                    textAlign = 'center';
                    x = marker.x - (cellWidth / 2);
                    break;
                case 'RIGHT':
                    textAlign = 'right';
                    x = marker.x - cellWidth;
                    break;
                case 'LEFT':
                default:
                    textAlign = 'left';
                    x = marker.x;
                    break;
            }
        } else {
            // Use smart alignment
            switch (alignmentInfo.alignment) {
                case 'CENTER':
                    textAlign = 'center';
                    x = marker.x - (cellWidth / 2);
                    break;
                case 'RIGHT':
                    textAlign = 'right';
                    x = marker.x - cellWidth;
                    break;
                case 'LEFT':
                default:
                    textAlign = 'left';
                    x = marker.x;
                    break;
            }
        }
        
        textElement.style.left = x + 'px';
        textElement.style.top = marker.y + 'px';
        textElement.style.width = cellWidth + 'px';
        textElement.style.textAlign = textAlign;
        
        // Add subtle shadow for better visibility on PDF background
        textElement.style.textShadow = '0 0 3px rgba(255,255,255,0.8), 0 0 6px rgba(255,255,255,0.6)';
        
        // Add slight letter spacing for better readability (like PDF rendering)
        textElement.style.letterSpacing = '0.5px';
        
        // Ensure crisp text rendering
        textElement.style.webkitFontSmoothing = 'antialiased';
        textElement.style.mozOsxFontSmoothing = 'grayscale';
        
        markerLayer.appendChild(textElement);
    });
    
    // Add preview overlay indicator
    const previewOverlay = document.createElement('div');
    previewOverlay.id = 'previewOverlay';
    previewOverlay.style.position = 'absolute';
    previewOverlay.style.top = '10px';
    previewOverlay.style.right = '10px';
    previewOverlay.style.background = 'rgba(40, 167, 69, 0.9)';
    previewOverlay.style.color = 'white';
    previewOverlay.style.padding = '8px 12px';
    previewOverlay.style.borderRadius = '4px';
    previewOverlay.style.fontSize = '12px';
    previewOverlay.style.fontWeight = 'bold';
    previewOverlay.style.zIndex = '1000';
    previewOverlay.style.boxShadow = '0 2px 8px rgba(0,0,0,0.3)';
    previewOverlay.textContent = '🎓 CERTIFICATE PREVIEW';
    
    markerLayer.appendChild(previewOverlay);
    
    // Add sample data info overlay
    const infoOverlay = document.createElement('div');
    infoOverlay.style.position = 'absolute';
    infoOverlay.style.bottom = '10px';
    infoOverlay.style.left = '10px';
    infoOverlay.style.background = 'rgba(0, 0, 0, 0.8)';
    infoOverlay.style.color = 'white';
    infoOverlay.style.padding = '6px 10px';
    infoOverlay.style.borderRadius = '4px';
    infoOverlay.style.fontSize = '11px';
    infoOverlay.style.zIndex = '1000';
    infoOverlay.style.maxWidth = '200px';
    infoOverlay.innerHTML = '<strong>Sample Data Preview</strong><br>Using Helvetica font as in final PDF';
    
    markerLayer.appendChild(infoOverlay);
}

function showEditingMode() {
    const canvas = document.getElementById('pdfCanvas');
    if (!canvas) return;
    
    const width = parseFloat(canvas.style.width);
    const height = parseFloat(canvas.style.height);
    
    // Restore normal editing view
    renderMarkers(currentPage, width, height);
}
</script>
<div id="globalLoaderOverlay" class="hidden"><div class="loader-spinner"></div></div>
</body>
</html>