<?php

/**
 * Batch Template Validation Script
 * 
 * This script validates all PDF templates in the templates directory
 * and generates a comprehensive report of their compatibility status.
 * 
 * Usage:
 *   php scripts/batch_validate_templates.php [options]
 * 
 * Options:
 *   --format=json|text|html    Output format (default: text)
 *   --output=filename          Save output to file
 *   --templates-dir=path       Templates directory (default: templates)
 *   --verbose                  Show detailed information
 *   --problems-only            Show only problematic templates
 */

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../config/config.php';

use App\Service\TemplateValidator;
use App\Service\ErrorHandler;
use App\Service\TemplateManagementService;

// Parse command line arguments
$options = getopt('', [
    'format:',
    'output:',
    'templates-dir:',
    'verbose',
    'problems-only',
    'help'
]);

if (isset($options['help'])) {
    showHelp();
    exit(0);
}

$format = $options['format'] ?? 'text';
$outputFile = $options['output'] ?? null;
$templatesDir = $options['templates-dir'] ?? 'templates';
$verbose = isset($options['verbose']);
$problemsOnly = isset($options['problems-only']);

// Validate format
if (!in_array($format, ['json', 'text', 'html'])) {
    echo "Error: Invalid format. Use json, text, or html.\n";
    exit(1);
}

// Check if templates directory exists
if (!is_dir($templatesDir)) {
    echo "Error: Templates directory not found: {$templatesDir}\n";
    exit(1);
}

try {
    // Initialize services
    $errorHandler = new ErrorHandler();
    $validator = new TemplateValidator($errorHandler);
    $managementService = new TemplateManagementService($validator, $errorHandler, $templatesDir);

    echo "Starting batch validation of templates in: {$templatesDir}\n";
    echo "Format: {$format}\n";
    if ($outputFile) {
        echo "Output will be saved to: {$outputFile}\n";
    }
    echo str_repeat('-', 50) . "\n";

    // Run batch validation
    $startTime = microtime(true);
    $results = $managementService->batchValidateTemplates();
    $endTime = microtime(true);

    $processingTime = round($endTime - $startTime, 2);

    // Get additional information if requested
    if ($verbose || $problemsOnly) {
        $problematicTemplates = $managementService->getProblematicTemplates();
        $recommendations = $managementService->generateReplacementRecommendations();
    }

    // Generate output based on format
    $output = '';
    
    switch ($format) {
        case 'json':
            $jsonData = [
                'validation_results' => $results,
                'processing_time' => $processingTime,
                'timestamp' => date('Y-m-d H:i:s')
            ];
            
            if ($verbose || $problemsOnly) {
                $jsonData['problematic_templates'] = $problematicTemplates ?? [];
                $jsonData['recommendations'] = $recommendations ?? [];
            }
            
            $output = json_encode($jsonData, JSON_PRETTY_PRINT);
            break;

        case 'html':
            $output = generateHtmlReport($results, $processingTime, $verbose ? ($problematicTemplates ?? []) : [], $verbose ? ($recommendations ?? []) : []);
            break;

        case 'text':
        default:
            $output = generateTextReport($results, $processingTime, $problemsOnly, $verbose ? ($problematicTemplates ?? []) : [], $verbose ? ($recommendations ?? []) : []);
            break;
    }

    // Output results
    if ($outputFile) {
        file_put_contents($outputFile, $output);
        echo "Results saved to: {$outputFile}\n";
    } else {
        echo $output;
    }

    // Exit with appropriate code
    exit($results['invalid_templates'] > 0 ? 1 : 0);

} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

function showHelp()
{
    echo "Batch Template Validation Script\n";
    echo "================================\n\n";
    echo "This script validates all PDF templates and generates compatibility reports.\n\n";
    echo "Usage:\n";
    echo "  php scripts/batch_validate_templates.php [options]\n\n";
    echo "Options:\n";
    echo "  --format=json|text|html    Output format (default: text)\n";
    echo "  --output=filename          Save output to file\n";
    echo "  --templates-dir=path       Templates directory (default: templates)\n";
    echo "  --verbose                  Show detailed information\n";
    echo "  --problems-only            Show only problematic templates\n";
    echo "  --help                     Show this help message\n\n";
    echo "Examples:\n";
    echo "  php scripts/batch_validate_templates.php\n";
    echo "  php scripts/batch_validate_templates.php --format=json --output=validation_report.json\n";
    echo "  php scripts/batch_validate_templates.php --verbose --problems-only\n";
}

function generateTextReport(array $results, float $processingTime, bool $problemsOnly, array $problematicTemplates, array $recommendations): string
{
    $output = '';
    
    // Header
    $output .= "PDF Template Validation Report\n";
    $output .= "==============================\n";
    $output .= "Generated: " . date('Y-m-d H:i:s') . "\n";
    $output .= "Processing time: {$processingTime} seconds\n\n";

    // Summary
    $output .= "Summary:\n";
    $output .= "--------\n";
    foreach ($results['summary'] as $summaryLine) {
        $output .= "• {$summaryLine}\n";
    }
    $output .= "\n";

    // Statistics
    $output .= "Statistics:\n";
    $output .= "-----------\n";
    $output .= "Total templates: {$results['total_templates']}\n";
    $output .= "Valid templates: {$results['valid_templates']}\n";
    $output .= "Invalid templates: {$results['invalid_templates']}\n";
    $output .= "FPDI compatible: {$results['fpdi_compatible']}\n";
    $output .= "FPDI incompatible: {$results['fpdi_incompatible']}\n\n";

    // Template details
    if (!$problemsOnly) {
        $output .= "Template Details:\n";
        $output .= "-----------------\n";
        
        foreach ($results['templates'] as $templateId => $template) {
            $output .= "Template: {$templateId}\n";
            $output .= "  File: {$template['filename']}\n";
            $output .= "  Size: " . formatFileSize($template['file_size']) . "\n";
            $output .= "  Status: {$template['status']}\n";
            $output .= "  Valid: " . ($template['is_valid'] ? 'Yes' : 'No') . "\n";
            $output .= "  FPDI Compatible: " . ($template['fpdi_compatible'] ? 'Yes' : 'No') . "\n";
            
            if (!empty($template['issues'])) {
                $output .= "  Issues:\n";
                foreach ($template['issues'] as $issue) {
                    $output .= "    - {$issue}\n";
                }
            }
            
            if (!empty($template['recommendations'])) {
                $output .= "  Recommendations:\n";
                foreach ($template['recommendations'] as $recommendation) {
                    $output .= "    - {$recommendation}\n";
                }
            }
            
            $output .= "\n";
        }
    }

    // Problematic templates
    if (!empty($problematicTemplates)) {
        $output .= "Problematic Templates:\n";
        $output .= "----------------------\n";
        
        foreach ($problematicTemplates as $templateId => $template) {
            $output .= "Template: {$templateId}\n";
            $output .= "  Issues: " . implode(', ', $template['issues']) . "\n";
            $output .= "  Recent errors: " . count($template['recent_errors']) . "\n";
            $output .= "\n";
        }
    }

    // Recommendations
    if (!empty($recommendations)) {
        $output .= "Recommendations:\n";
        $output .= "----------------\n";
        
        foreach ($recommendations as $recommendation) {
            $output .= "Template: {$recommendation['template_id']} (Priority: {$recommendation['priority']})\n";
            $output .= "  Issues: " . implode(', ', $recommendation['issues']) . "\n";
            $output .= "  Actions:\n";
            foreach ($recommendation['suggested_actions'] as $action) {
                $output .= "    - {$action}\n";
            }
            $output .= "\n";
        }
    }

    return $output;
}

function generateHtmlReport(array $results, float $processingTime, array $problematicTemplates, array $recommendations): string
{
    $html = '<!DOCTYPE html>
<html>
<head>
    <title>PDF Template Validation Report</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .header { background: #f5f5f5; padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .summary { background: #e8f5e8; padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .stats { display: flex; gap: 20px; margin-bottom: 20px; }
        .stat-box { background: #f0f8ff; padding: 15px; border-radius: 5px; text-align: center; flex: 1; }
        .template { border: 1px solid #ddd; margin: 10px 0; padding: 15px; border-radius: 5px; }
        .template.valid { border-left: 5px solid #4CAF50; }
        .template.invalid { border-left: 5px solid #f44336; }
        .template.incompatible { border-left: 5px solid #ff9800; }
        .issues { color: #d32f2f; }
        .recommendations { color: #1976d2; }
        table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
    </style>
</head>
<body>';

    $html .= '<div class="header">';
    $html .= '<h1>PDF Template Validation Report</h1>';
    $html .= '<p>Generated: ' . date('Y-m-d H:i:s') . '</p>';
    $html .= '<p>Processing time: ' . $processingTime . ' seconds</p>';
    $html .= '</div>';

    $html .= '<div class="summary">';
    $html .= '<h2>Summary</h2>';
    $html .= '<ul>';
    foreach ($results['summary'] as $summaryLine) {
        $html .= '<li>' . htmlspecialchars($summaryLine) . '</li>';
    }
    $html .= '</ul>';
    $html .= '</div>';

    $html .= '<div class="stats">';
    $html .= '<div class="stat-box"><h3>' . $results['total_templates'] . '</h3><p>Total Templates</p></div>';
    $html .= '<div class="stat-box"><h3>' . $results['valid_templates'] . '</h3><p>Valid Templates</p></div>';
    $html .= '<div class="stat-box"><h3>' . $results['fpdi_compatible'] . '</h3><p>FPDI Compatible</p></div>';
    $html .= '<div class="stat-box"><h3>' . $results['invalid_templates'] . '</h3><p>Invalid Templates</p></div>';
    $html .= '</div>';

    $html .= '<h2>Template Details</h2>';
    foreach ($results['templates'] as $templateId => $template) {
        $cssClass = $template['is_valid'] ? ($template['fpdi_compatible'] ? 'valid' : 'incompatible') : 'invalid';
        
        $html .= '<div class="template ' . $cssClass . '">';
        $html .= '<h3>' . htmlspecialchars($templateId) . '</h3>';
        $html .= '<p><strong>File:</strong> ' . htmlspecialchars($template['filename']) . '</p>';
        $html .= '<p><strong>Size:</strong> ' . formatFileSize($template['file_size']) . '</p>';
        $html .= '<p><strong>Status:</strong> ' . htmlspecialchars($template['status']) . '</p>';
        
        if (!empty($template['issues'])) {
            $html .= '<div class="issues"><strong>Issues:</strong><ul>';
            foreach ($template['issues'] as $issue) {
                $html .= '<li>' . htmlspecialchars($issue) . '</li>';
            }
            $html .= '</ul></div>';
        }
        
        if (!empty($template['recommendations'])) {
            $html .= '<div class="recommendations"><strong>Recommendations:</strong><ul>';
            foreach ($template['recommendations'] as $recommendation) {
                $html .= '<li>' . htmlspecialchars($recommendation) . '</li>';
            }
            $html .= '</ul></div>';
        }
        
        $html .= '</div>';
    }

    $html .= '</body></html>';
    
    return $html;
}

function formatFileSize(int $bytes): string
{
    if ($bytes === 0) return '0 B';
    
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = floor(log($bytes, 1024));
    
    return round($bytes / pow(1024, $i), 2) . ' ' . $units[$i];
}