<?php

namespace App\Service;

use Exception;
use setasign\Fpdi\PdfParser\CrossReference\CrossReferenceException;
use setasign\Fpdi\PdfParser\PdfParserException;

/**
 * Service for centralized error handling and logging
 */
class ErrorHandler implements ErrorHandlerInterface
{
    private string $logPath;
    private PdfErrorLogger $logger;

    public function __construct(string $logPath = 'logs/pdf_errors.log')
    {
        $this->logPath = $logPath;
        $this->logger = new PdfErrorLogger();
        $this->ensureLogDirectoryExists();
    }

    /**
     * Handle PDF processing errors
     */
    public function handlePdfError(Exception $exception, array $context): ErrorResponse
    {
        $templateName = $this->extractTemplateName($context);
        $errorType = $context['errorType'] ?? 'unknown';

        // Handle specific exception types
        if ($exception instanceof CrossReferenceException) {
            return $this->handleCompressionError($exception, $templateName, $context);
        }

        if ($exception instanceof PdfParserException) {
            return $this->handleParsingError($exception, $templateName, $context);
        }

        // Handle file not found errors
        if (strpos($exception->getMessage(), 'not found') !== false) {
            return ErrorResponse::processingError(
                "Template file not found",
                $exception->getMessage()
            );
        }

        // Handle general errors
        return $this->handleGeneralError($exception, $templateName, $context);
    }

    /**
     * Log PDF processing errors
     */
    public function logPdfError(string $templateId, string $error, array $context): void
    {
        // Use enhanced logger if we have an exception in context
        if (isset($context['exception']) && $context['exception'] instanceof Exception) {
            $templatePath = $context['templatePath'] ?? 'unknown';
            $this->logger->logError($templateId, $templatePath, $context['exception'], $context);
        }

        // Legacy logging for backward compatibility
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'template_id' => $templateId,
            'error' => $error,
            'context' => $context,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'Unknown'
        ];

        $logLine = json_encode($logEntry) . PHP_EOL;

        // Ensure log directory exists
        $this->ensureLogDirectoryExists();

        // Write to log file
        file_put_contents($this->logPath, $logLine, FILE_APPEND | LOCK_EX);

        // Log to file only (avoid interfering with PDF downloads)
        // error_log("PDF Processing Error - Template: {$templateId}, Error: {$error}");
    }

    /**
     * Handle compression compatibility errors
     */
    private function handleCompressionError(CrossReferenceException $exception, string $templateName, array $context): ErrorResponse
    {
        $userMessage = "The PDF template '{$templateName}' uses compression that is not supported by the free PDF processor. Please try a different template or contact support.";
        
        $technicalMessage = "FPDI CrossReference Exception: " . $exception->getMessage() . 
                          " (Template: " . ($context['templatePath'] ?? 'unknown') . ")";

        $suggestedActions = [
            'Try using a different PDF template from the available options',
            'Contact your administrator to replace this template with a compatible version',
            'Use the fallback generation option if available'
        ];

        return new ErrorResponse(
            $userMessage,
            $technicalMessage,
            $suggestedActions,
            'PDF_COMPRESSION_ERROR'
        );
    }

    /**
     * Handle PDF parsing errors
     */
    private function handleParsingError(PdfParserException $exception, string $templateName, array $context): ErrorResponse
    {
        $userMessage = "The PDF template '{$templateName}' could not be processed due to a parsing error. The file may be corrupted or in an unsupported format.";
        
        $technicalMessage = "FPDI Parser Exception: " . $exception->getMessage() . 
                          " (Template: " . ($context['templatePath'] ?? 'unknown') . ")";

        $suggestedActions = [
            'Verify the PDF file is not corrupted',
            'Try using a different PDF template',
            'Contact support if the problem persists'
        ];

        return new ErrorResponse(
            $userMessage,
            $technicalMessage,
            $suggestedActions,
            'PDF_PARSING_ERROR'
        );
    }

    /**
     * Handle general processing errors
     */
    private function handleGeneralError(Exception $exception, string $templateName, array $context): ErrorResponse
    {
        $userMessage = "Certificate generation failed for template '{$templateName}'. Please try again or contact support.";
        
        $technicalMessage = get_class($exception) . ": " . $exception->getMessage() . 
                          " (Template: " . ($context['templatePath'] ?? 'unknown') . ")";

        $suggestedActions = [
            'Try generating the certificate again',
            'Use a different template if available',
            'Contact support if the problem persists'
        ];

        return new ErrorResponse(
            $userMessage,
            $technicalMessage,
            $suggestedActions,
            'PDF_GENERAL_ERROR'
        );
    }

    /**
     * Extract template name from context
     */
    private function extractTemplateName(array $context): string
    {
        if (isset($context['templatePath'])) {
            return basename($context['templatePath'], '.pdf');
        }

        return 'Unknown Template';
    }

    /**
     * Ensure log directory exists
     */
    private function ensureLogDirectoryExists(): void
    {
        $logDir = dirname($this->logPath);
        
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
    }

    /**
     * Get recent error logs for a specific template
     */
    public function getRecentErrors(string $templateId, int $limit = 10): array
    {
        if (!file_exists($this->logPath)) {
            return [];
        }

        $lines = file($this->logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $errors = [];

        // Process lines in reverse order (most recent first)
        $lines = array_reverse($lines);

        foreach ($lines as $line) {
            $logEntry = json_decode($line, true);
            
            if ($logEntry && $logEntry['template_id'] === $templateId) {
                $errors[] = $logEntry;
                
                if (count($errors) >= $limit) {
                    break;
                }
            }
        }

        return $errors;
    }

    /**
     * Get error statistics
     */
    public function getErrorStats(int $days = 7): array
    {
        if (!file_exists($this->logPath)) {
            return ['total' => 0, 'by_template' => [], 'by_error_type' => []];
        }

        $lines = file($this->logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $stats = [
            'total' => 0,
            'by_template' => [],
            'by_error_type' => []
        ];

        foreach ($lines as $line) {
            $logEntry = json_decode($line, true);
            
            if ($logEntry && $logEntry['timestamp'] >= $cutoffDate) {
                $stats['total']++;
                
                // Count by template
                $templateId = $logEntry['template_id'];
                $stats['by_template'][$templateId] = ($stats['by_template'][$templateId] ?? 0) + 1;
                
                // Count by error type (extract from error message)
                $errorType = $this->categorizeError($logEntry['error']);
                $stats['by_error_type'][$errorType] = ($stats['by_error_type'][$errorType] ?? 0) + 1;
            }
        }

        return $stats;
    }

    /**
     * Categorize error for statistics
     */
    private function categorizeError(string $error): string
    {
        if (strpos($error, 'CrossReference') !== false) {
            return 'Compression Error';
        }
        
        if (strpos($error, 'Parser') !== false) {
            return 'Parsing Error';
        }
        
        if (strpos($error, 'not found') !== false) {
            return 'File Not Found';
        }
        
        if (strpos($error, 'Fallback') !== false) {
            return 'Fallback Error';
        }
        
        return 'Other';
    }

    /**
     * Log successful PDF processing
     */
    public function logSuccess(
        string $templateId,
        string $templatePath,
        string $processingMethod,
        float $processingTime,
        array $context = []
    ): void {
        $this->logger->logSuccess($templateId, $templatePath, $processingMethod, $processingTime, $context);
    }

    /**
     * Log template validation results
     */
    public function logValidation(
        string $templateId,
        string $templatePath,
        bool $isValid,
        array $validationResults,
        array $context = []
    ): void {
        $this->logger->logValidation($templateId, $templatePath, $isValid, $validationResults, $context);
    }

    /**
     * Get enhanced error statistics
     */
    public function getEnhancedErrorStats(int $days = 7): array
    {
        return $this->logger->getErrorStatistics($days);
    }

    /**
     * Get recent log entries from enhanced logger
     */
    public function getRecentLogEntries(int $limit = 100, string $level = null): array
    {
        return $this->logger->getRecentEntries($limit, $level);
    }

    /**
     * Get the enhanced logger instance
     */
    public function getLogger(): PdfErrorLogger
    {
        return $this->logger;
    }
}