<?php

namespace App\Service;

use TCPDF;

/**
 * Service for generating PDFs when FPDI fails - Universal PDF Support
 */
class FallbackPdfGenerator implements FallbackPdfGeneratorInterface
{
    private ErrorHandlerInterface $errorHandler;

    public function __construct(ErrorHandlerInterface $errorHandler)
    {
        $this->errorHandler = $errorHandler;
    }

    /**
     * Generate certificate from PDF using image conversion method
     * This method can handle ANY PDF by converting it to an image first
     */
    public function generateFromImageConversion(string $templatePath, array $fieldData): PdfResult
    {
        try {
            // Check if ImageMagick is available (preferred method for preserving original design)
            if (!extension_loaded('imagick')) {
                // Try alternative image conversion methods
                return $this->tryAlternativeImageConversion($templatePath, $fieldData);
            }

            // Use ImageMagick to preserve the original template design
            return $this->generateFromImageMagick($templatePath, $fieldData);

        } catch (\Exception $e) {
            // Re-throw to try next strategy (no output to avoid interfering with PDF download)
            throw $e;
        }
    }

    /**
     * Try alternative methods to convert PDF when ImageMagick is not available
     */
    private function tryAlternativeImageConversion(string $templatePath, array $fieldData): PdfResult
    {
        // Try using system commands if available
        if ($this->canUseSystemImageMagick()) {
            return $this->generateUsingSystemImageMagick($templatePath, $fieldData);
        }
        
        // Try using GhostScript if available
        if ($this->canUseGhostScript()) {
            return $this->generateUsingGhostScript($templatePath, $fieldData);
        }
        
        // If no image conversion available, throw exception to try next strategy
        throw new \Exception('No image conversion method available');
    }

    /**
     * Check if system ImageMagick command is available
     */
    private function canUseSystemImageMagick(): bool
    {
        $output = [];
        $return_var = 0;
        exec('magick -version 2>&1', $output, $return_var);
        return $return_var === 0;
    }

    /**
     * Check if GhostScript is available
     */
    private function canUseGhostScript(): bool
    {
        $output = [];
        $return_var = 0;
        exec('gs -version 2>&1', $output, $return_var);
        return $return_var === 0;
    }

    /**
     * Generate using system ImageMagick command
     */
    private function generateUsingSystemImageMagick(string $templatePath, array $fieldData): PdfResult
    {
        try {
            // Create temporary image file
            $tempImagePath = tempnam(sys_get_temp_dir(), 'cert_template_') . '.png';
            
            // Convert PDF to image using system command
            $command = sprintf(
                'magick -density 300 "%s[0]" "%s" 2>&1',
                escapeshellarg($templatePath),
                escapeshellarg($tempImagePath)
            );
            
            $output = [];
            $return_var = 0;
            exec($command, $output, $return_var);
            
            if ($return_var !== 0 || !file_exists($tempImagePath)) {
                throw new \Exception('System ImageMagick conversion failed');
            }
            
            // Get image dimensions
            $imageInfo = getimagesize($tempImagePath);
            if (!$imageInfo) {
                unlink($tempImagePath);
                throw new \Exception('Could not get image dimensions');
            }
            
            $width = $imageInfo[0];
            $height = $imageInfo[1];
            
            // Convert to PDF dimensions (assuming 300 DPI)
            $pdfWidth = ($width / 300) * 25.4; // Convert to mm
            $pdfHeight = ($height / 300) * 25.4; // Convert to mm
            
            // Create new PDF with original template as background
            $pdf = new TCPDF(($pdfWidth > $pdfHeight) ? 'L' : 'P', 'mm', [$pdfWidth, $pdfHeight], true, 'UTF-8', false);
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            $pdf->SetMargins(0, 0, 0);
            $pdf->SetAutoPageBreak(false);
            $pdf->AddPage();
            
            // Add the original template as background
            $pdf->Image($tempImagePath, 0, 0, $pdfWidth, $pdfHeight, 'PNG');
            
            // Overlay user data on the original template
            $this->overlayFieldsOnOriginalTemplate($pdf, $fieldData, $pdfWidth, $pdfHeight, $width, $height);
            
            // Clean up
            unlink($tempImagePath);
            
            $pdfContent = $pdf->Output('', 'S');
            
            return PdfResult::success($pdfContent, 'system_imagemagick', [
                'originalDimensions' => ['width' => $width, 'height' => $height],
                'pdfDimensions' => ['width' => $pdfWidth, 'height' => $pdfHeight],
                'fieldCount' => count($fieldData['field_values'] ?? []),
                'preservedOriginalDesign' => true
            ]);
            
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Generate using GhostScript
     */
    private function generateUsingGhostScript(string $templatePath, array $fieldData): PdfResult
    {
        try {
            // Create temporary image file
            $tempImagePath = tempnam(sys_get_temp_dir(), 'cert_template_') . '.png';
            
            // Convert PDF to image using GhostScript
            $command = sprintf(
                'gs -dNOPAUSE -dBATCH -sDEVICE=png16m -r300 -dFirstPage=1 -dLastPage=1 -sOutputFile="%s" "%s" 2>&1',
                escapeshellarg($tempImagePath),
                escapeshellarg($templatePath)
            );
            
            $output = [];
            $return_var = 0;
            exec($command, $output, $return_var);
            
            if ($return_var !== 0 || !file_exists($tempImagePath)) {
                throw new \Exception('GhostScript conversion failed');
            }
            
            // Get image dimensions
            $imageInfo = getimagesize($tempImagePath);
            if (!$imageInfo) {
                unlink($tempImagePath);
                throw new \Exception('Could not get image dimensions');
            }
            
            $width = $imageInfo[0];
            $height = $imageInfo[1];
            
            // Convert to PDF dimensions (assuming 300 DPI)
            $pdfWidth = ($width / 300) * 25.4;
            $pdfHeight = ($height / 300) * 25.4;
            
            // Create new PDF with original template as background
            $pdf = new TCPDF(($pdfWidth > $pdfHeight) ? 'L' : 'P', 'mm', [$pdfWidth, $pdfHeight], true, 'UTF-8', false);
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            $pdf->SetMargins(0, 0, 0);
            $pdf->SetAutoPageBreak(false);
            $pdf->AddPage();
            
            // Add the original template as background
            $pdf->Image($tempImagePath, 0, 0, $pdfWidth, $pdfHeight, 'PNG');
            
            // Overlay user data
            $this->overlayFieldsOnOriginalTemplate($pdf, $fieldData, $pdfWidth, $pdfHeight, $width, $height);
            
            // Clean up
            unlink($tempImagePath);
            
            $pdfContent = $pdf->Output('', 'S');
            
            return PdfResult::success($pdfContent, 'ghostscript', [
                'originalDimensions' => ['width' => $width, 'height' => $height],
                'pdfDimensions' => ['width' => $pdfWidth, 'height' => $pdfHeight],
                'fieldCount' => count($fieldData['field_values'] ?? []),
                'preservedOriginalDesign' => true
            ]);
            
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Recreate template structure preserving original design intent
     */
    public function recreateTemplateStructure(string $templatePath, array $fieldData, array $templateMetadata): PdfResult
    {
        try {
            // Try to extract template dimensions if possible
            $templateDimensions = $this->extractTemplateDimensions($templatePath);
            
            // Create PDF with original dimensions or default to A4 landscape
            $orientation = $templateDimensions['orientation'] ?? 'L';
            $pageSize = $templateDimensions['size'] ?? 'A4';
            
            $pdf = new TCPDF($orientation, 'mm', $pageSize, true, 'UTF-8', false);
            
            // Set document information
            $pdf->SetCreator('Certificate System - Template Recreation');
            $pdf->SetAuthor('Certificate Generator');
            $pdf->SetTitle('Certificate - ' . ($templateMetadata['templateName'] ?? 'Template'));
            
            // Remove default header/footer
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            
            // Set margins to match original template style
            $pdf->SetMargins(10, 10, 10);
            $pdf->SetAutoPageBreak(false);
            
            // Add a page
            $pdf->AddPage();
            
            // Create sophisticated template background that mimics original
            $this->createSophisticatedTemplateBackground($pdf, $templateMetadata, $templateDimensions);
            
            // Add field data with coordinate-aware positioning
            $this->addFieldsWithCoordinateAwarePositioning($pdf, $fieldData, $templateMetadata);
            
            // Get PDF content
            $pdfContent = $pdf->Output('', 'S');
            
            return PdfResult::success($pdfContent, 'template_recreation', [
                'templateMetadata' => $templateMetadata,
                'fieldCount' => count($fieldData['field_values'] ?? []),
                'preservedOriginalLayout' => true,
                'templateDimensions' => $templateDimensions
            ]);
            
        } catch (\Exception $e) {
            // Re-throw to try next strategy (no output to avoid interfering with PDF download)
            throw $e;
        }
    }

    /**
     * Extract template dimensions and characteristics
     */
    private function extractTemplateDimensions(string $templatePath): array
    {
        $dimensions = [
            'orientation' => 'L', // Default landscape
            'size' => 'A4',
            'width' => 297,
            'height' => 210
        ];
        
        // Try to get basic file info
        if (file_exists($templatePath)) {
            $fileSize = filesize($templatePath);
            
            // Heuristic: larger files often indicate more complex/larger templates
            if ($fileSize > 2000000) { // > 2MB
                $dimensions['size'] = [420, 297]; // A3 landscape
                $dimensions['width'] = 420;
                $dimensions['height'] = 297;
            }
        }
        
        return $dimensions;
    }

    /**
     * Create sophisticated background that mimics original template
     */
    private function createSophisticatedTemplateBackground(TCPDF $pdf, array $templateMetadata, array $templateDimensions): void
    {
        $templateName = strtolower($templateMetadata['templateName'] ?? '');
        
        // Set background
        $pdf->SetFillColor(255, 255, 255);
        $pdf->Rect(0, 0, $pdf->getPageWidth(), $pdf->getPageHeight(), 'F');
        
        // Create borders that look more like real certificates
        $this->createRealisticCertificateBorder($pdf);
        
        // Add template-specific elements based on name analysis
        if (strpos($templateName, 'completion') !== false || strpos($templateName, 'course') !== false) {
            $this->addCourseCompletionElements($pdf);
        } elseif (strpos($templateName, 'achievement') !== false || strpos($templateName, 'award') !== false) {
            $this->addAchievementElements($pdf);
        } elseif (strpos($templateName, 'participation') !== false) {
            $this->addParticipationElements($pdf);
        } else {
            $this->addGenericProfessionalElements($pdf);
        }
    }

    /**
     * Add fields with coordinate-aware positioning
     */
    private function addFieldsWithCoordinateAwarePositioning(TCPDF $pdf, array $fieldData, array $templateMetadata): void
    {
        $fieldValues = $fieldData['field_values'] ?? [];
        $fieldCoordinates = $fieldData['field_coordinates'] ?? [];
        
        // For template recreation, coordinates from the original template may not align
        // with our recreated background, so use intelligent positioning instead
        if (!empty($fieldCoordinates)) {
            // Try to use coordinates but with fallback to intelligent positioning
            $this->addFieldsWithCoordinatesFallback($pdf, $fieldValues, $fieldCoordinates, $templateMetadata);
        } else {
            // Use advanced intelligent positioning
            $this->addFieldsWithAdvancedPositioning($pdf, $fieldValues, $templateMetadata);
        }
    }

    /**
     * Generate using ImageMagick (handles ANY PDF) - Preserves Original Template Design
     */
    private function generateFromImageMagick(string $templatePath, array $fieldData): PdfResult
    {
        try {
            // Convert PDF first page to high-resolution image
            $imagick = new \Imagick();
            $imagick->setResolution(300, 300); // High resolution for quality
            $imagick->readImage($templatePath . '[0]'); // First page only
            $imagick->setImageFormat('png');
            
            // Get image dimensions
            $width = $imagick->getImageWidth();
            $height = $imagick->getImageHeight();
            
            // Convert to PDF dimensions (assuming 300 DPI)
            $pdfWidth = ($width / 300) * 25.4; // Convert to mm
            $pdfHeight = ($height / 300) * 25.4; // Convert to mm
            
            // Create new PDF with exact dimensions of original
            $pdf = new TCPDF(($pdfWidth > $pdfHeight) ? 'L' : 'P', 'mm', [$pdfWidth, $pdfHeight], true, 'UTF-8', false);
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            $pdf->SetMargins(0, 0, 0);
            $pdf->SetAutoPageBreak(false);
            $pdf->AddPage();
            
            // Add the original template as background image
            $tempImagePath = tempnam(sys_get_temp_dir(), 'cert_template_') . '.png';
            $imagick->writeImage($tempImagePath);
            $pdf->Image($tempImagePath, 0, 0, $pdfWidth, $pdfHeight, 'PNG');
            
            // Overlay user data with proper image conversion scaling
            $this->overlayFieldsOnImageConvertedTemplate($pdf, $fieldData, $pdfWidth, $pdfHeight, $width, $height);
            
            // Clean up
            $imagick->clear();
            unlink($tempImagePath);
            
            $pdfContent = $pdf->Output('', 'S');
            
            return PdfResult::success($pdfContent, 'image_conversion', [
                'originalDimensions' => ['width' => $width, 'height' => $height],
                'pdfDimensions' => ['width' => $pdfWidth, 'height' => $pdfHeight],
                'fieldCount' => count($fieldData['field_values'] ?? []),
                'preservedOriginalDesign' => true
            ]);
            
        } catch (\Exception $e) {
            // Re-throw to try next strategy (no output to avoid interfering with PDF download)
            throw $e;
        }
    }

    /**
     * Generate certificate from incompatible template using TCPDF only
     */
    public function generateFromIncompatibleTemplate(string $templatePath, array $fieldData): PdfResult
    {
        try {
            // Extract template information for fallback generation
            $templateInfo = $this->extractTemplateInfo($templatePath);
            
            // Create blank certificate with template styling
            return $this->createBlankCertificate($fieldData, $templateInfo);
            
        } catch (\Exception $e) {
            $this->errorHandler->logPdfError(
                basename($templatePath),
                'Fallback generation failed: ' . $e->getMessage(),
                ['templatePath' => $templatePath, 'fieldData' => array_keys($fieldData)]
            );
            
            return PdfResult::failure(
                'Unable to generate certificate using fallback method',
                [
                    'errorCode' => 'FALLBACK_GENERATION_FAILED',
                    'technicalMessage' => $e->getMessage(),
                    'suggestedActions' => [
                        'Contact administrator to replace this template',
                        'Try using a different template',
                        'Ensure the template file is not corrupted'
                    ]
                ]
            );
        }
    }

    /**
     * Create blank certificate with field data
     */
    public function createBlankCertificate(array $fieldData, array $templateInfo): PdfResult
    {
        try {
            // Create new TCPDF instance
            $pdf = new TCPDF('L', 'mm', 'A4', true, 'UTF-8', false);
            
            // Set document information
            $pdf->SetCreator('Certificate System');
            $pdf->SetAuthor('Certificate Generator');
            $pdf->SetTitle('Certificate');
            
            // Remove default header/footer
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            
            // Set margins
            $pdf->SetMargins(20, 20, 20);
            $pdf->SetAutoPageBreak(false);
            
            // Add a page
            $pdf->AddPage();
            
            // Set background color
            $pdf->SetFillColor(255, 255, 255);
            $pdf->Rect(0, 0, $pdf->getPageWidth(), $pdf->getPageHeight(), 'F');
            
            // Add decorative border
            $this->addCertificateBorder($pdf);
            
            // Add certificate title
            $this->addCertificateTitle($pdf, $templateInfo);
            
            // Add field data to certificate
            $this->addFieldsToBlankCertificate($pdf, $fieldData);
            
            // Add signature area
            $this->addSignatureArea($pdf);
            
            // Get PDF content
            $pdfContent = $pdf->Output('', 'S');
            
            return PdfResult::success($pdfContent, 'blank', [
                'templateInfo' => $templateInfo,
                'fieldCount' => count($fieldData['field_values'] ?? [])
            ]);
            
        } catch (\Exception $e) {
            $this->errorHandler->logPdfError(
                'blank_certificate',
                'Blank certificate generation failed: ' . $e->getMessage(),
                ['fieldData' => array_keys($fieldData)]
            );
            
            return PdfResult::failure(
                'Unable to generate blank certificate',
                [
                    'errorCode' => 'BLANK_CERT_GENERATION_FAILED',
                    'technicalMessage' => $e->getMessage(),
                    'suggestedActions' => [
                        'Check that all required fields are provided',
                        'Contact administrator if the problem persists'
                    ]
                ]
            );
        }
    }

    /**
     * Extract basic information from template file
     */
    private function extractTemplateInfo(string $templatePath): array
    {
        $templateInfo = [
            'name' => basename($templatePath, '.pdf'),
            'width' => 297, // A4 landscape width in mm
            'height' => 210, // A4 landscape height in mm
            'orientation' => 'L'
        ];
        
        // Try to get file size for metadata
        if (file_exists($templatePath)) {
            $templateInfo['fileSize'] = filesize($templatePath);
            $templateInfo['lastModified'] = filemtime($templatePath);
        }
        
        return $templateInfo;
    }

    /**
     * Add decorative border to certificate
     */
    private function addCertificateBorder(TCPDF $pdf): void
    {
        // Outer decorative border
        $pdf->SetLineWidth(3);
        $pdf->SetDrawColor(0, 50, 100);
        $pdf->Rect(8, 8, $pdf->getPageWidth() - 16, $pdf->getPageHeight() - 16);
        
        // Inner border
        $pdf->SetLineWidth(1);
        $pdf->SetDrawColor(100, 0, 0);
        $pdf->Rect(12, 12, $pdf->getPageWidth() - 24, $pdf->getPageHeight() - 24);
        
        // Decorative corner elements
        $pdf->SetLineWidth(0.5);
        $pdf->SetDrawColor(150, 150, 150);
        
        // Top corners
        $pdf->Line(20, 20, 40, 20);
        $pdf->Line(20, 20, 20, 40);
        $pdf->Line($pdf->getPageWidth() - 40, 20, $pdf->getPageWidth() - 20, 20);
        $pdf->Line($pdf->getPageWidth() - 20, 20, $pdf->getPageWidth() - 20, 40);
        
        // Bottom corners
        $pdf->Line(20, $pdf->getPageHeight() - 20, 40, $pdf->getPageHeight() - 20);
        $pdf->Line(20, $pdf->getPageHeight() - 40, 20, $pdf->getPageHeight() - 20);
        $pdf->Line($pdf->getPageWidth() - 40, $pdf->getPageHeight() - 20, $pdf->getPageWidth() - 20, $pdf->getPageHeight() - 20);
        $pdf->Line($pdf->getPageWidth() - 20, $pdf->getPageHeight() - 40, $pdf->getPageWidth() - 20, $pdf->getPageHeight() - 20);
    }

    /**
     * Add certificate title with professional styling
     */
    private function addCertificateTitle(TCPDF $pdf, array $templateInfo): void
    {
        // Main title with elegant styling
        $pdf->SetFont('helvetica', 'B', 28);
        $pdf->SetTextColor(0, 50, 100);
        $pdf->SetXY(20, 30);
        $pdf->Cell($pdf->getPageWidth() - 40, 20, 'CERTIFICATE OF ACHIEVEMENT', 0, 1, 'C');
        
        // Decorative line under title
        $pdf->SetLineWidth(1);
        $pdf->SetDrawColor(100, 0, 0);
        $centerX = $pdf->getPageWidth() / 2;
        $pdf->Line($centerX - 60, 55, $centerX + 60, 55);
        
        // Subtitle
        $pdf->SetFont('helvetica', '', 14);
        $pdf->SetTextColor(0, 0, 0);
        $pdf->SetXY(20, 65);
        $pdf->Cell($pdf->getPageWidth() - 40, 10, 'This is to certify that', 0, 1, 'C');
    }

    /**
     * Add fields to blank certificate with coordinate-aware layout
     */
    private function addFieldsToBlankCertificate(TCPDF $pdf, array $fieldData): void
    {
        $fieldValues = $fieldData['field_values'] ?? [];
        $fieldCoordinates = $fieldData['field_coordinates'] ?? [];
        
        // If coordinates are available, use them for precise positioning
        if (!empty($fieldCoordinates)) {
            $this->addFieldsWithCoordinates($pdf, $fieldValues, $fieldCoordinates);
            return;
        }
        
        // Fallback to professional layout if no coordinates
        $yPosition = 80;
        
        // Main recipient name (prominent display)
        if (isset($fieldValues['name'])) {
            $pdf->SetFont('helvetica', 'B', 24);
            $pdf->SetTextColor(0, 0, 100);
            $pdf->SetXY(20, $yPosition);
            $pdf->Cell($pdf->getPageWidth() - 40, 20, $fieldValues['name'], 0, 1, 'C');
            $yPosition += 35;
        }
        
        // Course/Achievement
        if (isset($fieldValues['course'])) {
            $pdf->SetFont('helvetica', '', 14);
            $pdf->SetTextColor(0, 0, 0);
            $pdf->SetXY(20, $yPosition);
            $pdf->Cell($pdf->getPageWidth() - 40, 12, 'has successfully completed', 0, 1, 'C');
            $yPosition += 20;
            
            $pdf->SetFont('helvetica', 'B', 18);
            $pdf->SetTextColor(100, 0, 0);
            $pdf->SetXY(20, $yPosition);
            $pdf->Cell($pdf->getPageWidth() - 40, 15, $fieldValues['course'], 0, 1, 'C');
            $yPosition += 30;
        }
        
        // Additional fields in a structured layout
        $remainingFields = array_diff_key($fieldValues, ['name' => '', 'course' => '']);
        
        if (!empty($remainingFields)) {
            $pdf->SetFont('helvetica', '', 12);
            $pdf->SetTextColor(0, 0, 0);
            
            $leftColumn = [];
            $rightColumn = [];
            $fieldIndex = 0;
            
            foreach ($remainingFields as $fieldName => $fieldValue) {
                if ($fieldIndex % 2 == 0) {
                    $leftColumn[$fieldName] = $fieldValue;
                } else {
                    $rightColumn[$fieldName] = $fieldValue;
                }
                $fieldIndex++;
            }
            
            // Display fields in two columns
            $maxRows = max(count($leftColumn), count($rightColumn));
            $startY = $yPosition;
            
            $leftFields = array_keys($leftColumn);
            $rightFields = array_keys($rightColumn);
            
            for ($i = 0; $i < $maxRows; $i++) {
                $currentY = $startY + ($i * 20);
                
                // Left column
                if (isset($leftFields[$i])) {
                    $fieldName = $leftFields[$i];
                    $fieldValue = $leftColumn[$fieldName];
                    
                    $pdf->SetFont('helvetica', '', 11);
                    $pdf->SetXY(40, $currentY);
                    $pdf->Cell(60, 10, ucfirst(str_replace('_', ' ', $fieldName)) . ':', 0, 0, 'L');
                    
                    $pdf->SetFont('helvetica', 'B', 11);
                    $pdf->SetXY(105, $currentY);
                    $pdf->Cell(80, 10, $fieldValue, 0, 0, 'L');
                }
                
                // Right column
                if (isset($rightFields[$i])) {
                    $fieldName = $rightFields[$i];
                    $fieldValue = $rightColumn[$fieldName];
                    
                    $pdf->SetFont('helvetica', '', 11);
                    $pdf->SetXY(200, $currentY);
                    $pdf->Cell(60, 10, ucfirst(str_replace('_', ' ', $fieldName)) . ':', 0, 0, 'L');
                    
                    $pdf->SetFont('helvetica', 'B', 11);
                    $pdf->SetXY(265, $currentY);
                    $pdf->Cell(80, 10, $fieldValue, 0, 0, 'L');
                }
            }
        }
    }

    /**
     * Add professional signature area to certificate
     */
    private function addSignatureArea(TCPDF $pdf): void
    {
        $bottomY = $pdf->getPageHeight() - 60;
        
        // Date section
        $pdf->SetFont('helvetica', '', 11);
        $pdf->SetTextColor(0, 0, 0);
        $pdf->SetXY(50, $bottomY);
        $pdf->Cell(80, 8, 'Date: ' . date('F j, Y'), 0, 0, 'L');
        
        // Signature section
        $pdf->SetXY($pdf->getPageWidth() - 150, $bottomY);
        $pdf->Cell(80, 8, 'Authorized Signature', 0, 0, 'C');
        
        // Signature line
        $pdf->SetLineWidth(0.5);
        $pdf->SetDrawColor(0, 0, 0);
        $pdf->Line($pdf->getPageWidth() - 150, $bottomY + 20, $pdf->getPageWidth() - 70, $bottomY + 20);
        
        // Authority title
        $pdf->SetFont('helvetica', '', 9);
        $pdf->SetXY($pdf->getPageWidth() - 150, $bottomY + 25);
        $pdf->Cell(80, 6, 'Director of Education', 0, 0, 'C');
        
        // Institution name/seal area
        $pdf->SetFont('helvetica', 'B', 10);
        $pdf->SetTextColor(0, 50, 100);
        $pdf->SetXY(50, $bottomY + 20);
        $pdf->Cell(120, 8, 'CertifyEd Learning Platform', 0, 0, 'L');
        
        $pdf->SetFont('helvetica', '', 8);
        $pdf->SetTextColor(100, 100, 100);
        $pdf->SetXY(50, $bottomY + 30);
        $pdf->Cell(120, 6, 'Professional Development & Certification', 0, 0, 'L');
    }

    /**
     * Create template-like background for recreated templates
     */
    private function createTemplateBackground(TCPDF $pdf, array $templateMetadata): void
    {
        // Set background color
        $pdf->SetFillColor(255, 255, 255);
        $pdf->Rect(0, 0, $pdf->getPageWidth(), $pdf->getPageHeight(), 'F');
        
        // Add decorative elements based on template name
        $templateName = strtolower($templateMetadata['templateName'] ?? '');
        
        if (strpos($templateName, 'student') !== false) {
            $this->addStudentTemplateElements($pdf);
        } elseif (strpos($templateName, 'teacher') !== false) {
            $this->addTeacherTemplateElements($pdf);
        } else {
            $this->addGenericTemplateElements($pdf);
        }
    }

    /**
     * Add fields with intelligent positioning based on template metadata
     */
    private function addFieldsWithIntelligentPositioning(TCPDF $pdf, array $fieldData, array $templateMetadata): void
    {
        $fieldValues = $fieldData['field_values'] ?? [];
        $fieldCoordinates = $fieldData['field_coordinates'] ?? [];
        
        // If we have coordinates, use them
        if (!empty($fieldCoordinates)) {
            $this->addFieldsWithCoordinates($pdf, $fieldValues, $fieldCoordinates);
        } else {
            // Use intelligent positioning based on field types
            $this->addFieldsWithSmartPositioning($pdf, $fieldValues, $templateMetadata);
        }
    }

    /**
     * Overlay fields on image-converted template with proper scaling
     */
    private function overlayFieldsOnImageConvertedTemplate(TCPDF $pdf, array $fieldData, float $pdfWidth, float $pdfHeight, int $imageWidth, int $imageHeight): void
    {
        $fieldValues = $fieldData['field_values'] ?? [];
        $fieldCoordinates = $fieldData['field_coordinates'] ?? [];
        
        if (!empty($fieldCoordinates)) {
            foreach ($fieldValues as $fieldName => $fieldValue) {
                if (empty($fieldValue)) continue;
                
                // Find coordinates for this field
                $coord = null;
                foreach ($fieldCoordinates as $c) {
                    if (strtolower($c['type'] ?? '') === strtolower($fieldName)) {
                        $coord = $c;
                        break;
                    }
                }
                
                if ($coord) {
                    // For image conversion, the PDF dimensions match the original template
                    // So we need to scale from preview dimensions to the actual template dimensions
                    $x = $coord['x'] ?? 0;
                    $y = $coord['y'] ?? 0;
                    
                    // Scale from preview to actual template dimensions
                    if (!empty($coord['preview_width']) && !empty($coord['preview_height'])) {
                        // The image dimensions represent the actual template at 300 DPI
                        // Convert image pixels to PDF points (1 inch = 72 points, 300 DPI)
                        $templateWidthPoints = ($imageWidth / 300) * 72;
                        $templateHeightPoints = ($imageHeight / 300) * 72;
                        
                        // Scale coordinates from preview to template points
                        $scaleX = $templateWidthPoints / $coord['preview_width'];
                        $scaleY = $templateHeightPoints / $coord['preview_height'];
                        $x = $x * $scaleX;
                        $y = $y * $scaleY;
                        
                        // Convert points to mm for TCPDF (1 point = 0.352778 mm)
                        $x = $x * 0.352778;
                        $y = $y * 0.352778;
                    }
                    
                    $fontSize = max($coord['font_size'] ?? 14, 12); // Minimum 12pt
                    $fontColor = $coord['font_color'] ?? '#000000';
                    
                    // Set font and color for clean text
                    $pdf->SetFont('helvetica', 'B', $fontSize);
                    $this->setTextColorFromHex($pdf, $fontColor);
                    
                    // Position and add text without background
                    $pdf->SetXY($x, $y);
                    $pdf->Cell(0, 10, $fieldValue, 0, 0, 'L');
                }
            }
        } else {
            // Use intelligent positioning if no coordinates
            $this->overlayFieldsWithIntelligentPositioning($pdf, $fieldValues, $pdfWidth, $pdfHeight);
        }
    }

    /**
     * Intelligently overlay fields on the original template design
     */
    private function overlayFieldsOnOriginalTemplate(TCPDF $pdf, array $fieldData, float $pdfWidth, float $pdfHeight, int $imageWidth, int $imageHeight): void
    {
        $fieldValues = $fieldData['field_values'] ?? [];
        $fieldCoordinates = $fieldData['field_coordinates'] ?? [];
        
        // If we have specific coordinates, use them (scaled to the actual template size)
        if (!empty($fieldCoordinates)) {
            $this->overlayFieldsWithCoordinates($pdf, $fieldValues, $fieldCoordinates, $pdfWidth, $pdfHeight);
        } else {
            // Use intelligent positioning based on template analysis
            $this->overlayFieldsWithIntelligentPositioning($pdf, $fieldValues, $pdfWidth, $pdfHeight);
        }
    }

    /**
     * Overlay fields using provided coordinates (scaled to actual template)
     */
    private function overlayFieldsWithCoordinates(TCPDF $pdf, array $fieldValues, array $fieldCoordinates, float $pdfWidth, float $pdfHeight): void
    {
        foreach ($fieldValues as $fieldName => $fieldValue) {
            if (empty($fieldValue)) continue;
            
            // Find coordinates for this field
            $coord = null;
            foreach ($fieldCoordinates as $c) {
                if (strtolower($c['type'] ?? '') === strtolower($fieldName)) {
                    $coord = $c;
                    break;
                }
            }
            
            if ($coord) {
                // Get coordinates - check both formats
                $x = $coord['pdf_x'] ?? $coord['x'] ?? 0;
                $y = $coord['pdf_y'] ?? $coord['y'] ?? 0;
                
                // Scale coordinates from preview dimensions to PDF dimensions if needed
                if (!empty($coord['preview_width']) && !empty($coord['preview_height'])) {
                    $scaleX = $pdfWidth / $coord['preview_width'];
                    $scaleY = $pdfHeight / $coord['preview_height'];
                    $x = $x * $scaleX;
                    $y = $y * $scaleY;
                }
                
                $fontSize = $coord['font_size'] ?? 14;
                $fontColor = $coord['font_color'] ?? '#000000';
                
                // Apply intelligent text alignment
                $alignmentResult = $this->calculateIntelligentTextAlignment($fieldName, $fieldValue, $x, $y, $pdfWidth, $pdfHeight, $coord);
                
                // Set font and color - ensure visibility
                $pdf->SetFont('helvetica', 'B', max($fontSize, 14)); // Minimum 14pt for visibility
                $this->setTextColorFromHex($pdf, $fontColor);
                
                // Position and add text with intelligent alignment
                $pdf->SetXY($alignmentResult['x'], $alignmentResult['y']);
                $pdf->Cell($alignmentResult['width'], 10, $fieldValue, 0, 0, $alignmentResult['align']);
            }
        }
    }

    /**
     * Overlay fields using intelligent positioning when no coordinates are available
     */
    private function overlayFieldsWithIntelligentPositioning(TCPDF $pdf, array $fieldValues, float $pdfWidth, float $pdfHeight): void
    {
        // Analyze template layout and position fields intelligently
        $centerX = $pdfWidth / 2;
        $centerY = $pdfHeight / 2;
        
        // Common certificate field positions (as percentages of template size)
        $fieldPositions = [
            'name' => ['x' => 0.5, 'y' => 0.45, 'size' => 18, 'align' => 'C', 'color' => '#000080'],
            'course' => ['x' => 0.5, 'y' => 0.55, 'size' => 14, 'align' => 'C', 'color' => '#800000'],
            'date' => ['x' => 0.25, 'y' => 0.8, 'size' => 11, 'align' => 'L', 'color' => '#000000'],
            'instructor' => ['x' => 0.75, 'y' => 0.8, 'size' => 11, 'align' => 'R', 'color' => '#000000'],
            'duration' => ['x' => 0.25, 'y' => 0.75, 'size' => 10, 'align' => 'L', 'color' => '#666666'],
            'grade' => ['x' => 0.75, 'y' => 0.75, 'size' => 12, 'align' => 'R', 'color' => '#008000'],
        ];
        
        foreach ($fieldValues as $fieldName => $fieldValue) {
            if (empty($fieldValue)) continue;
            
            // Get position for this field type, or use default
            $position = $fieldPositions[$fieldName] ?? ['x' => 0.5, 'y' => 0.6, 'size' => 12, 'align' => 'C', 'color' => '#000000'];
            
            // Calculate actual coordinates
            $x = $pdfWidth * $position['x'];
            $y = $pdfHeight * $position['y'];
            
            // Set font and styling
            $pdf->SetFont('helvetica', 'B', $position['size']);
            $this->setTextColorFromHex($pdf, $position['color']);
            
            // Position text based on alignment
            if ($position['align'] === 'C') {
                $pdf->SetXY($x - 50, $y); // Center alignment
                $pdf->Cell(100, 10, $fieldValue, 0, 0, 'C');
            } elseif ($position['align'] === 'R') {
                $pdf->SetXY($x - 100, $y); // Right alignment
                $pdf->Cell(100, 10, $fieldValue, 0, 0, 'R');
            } else {
                $pdf->SetXY($x, $y); // Left alignment
                $pdf->Cell(100, 10, $fieldValue, 0, 0, 'L');
            }
        }
    }

    /**
     * Calculate intelligent text alignment based on field type and position
     */
    private function calculateIntelligentTextAlignment(string $fieldType, string $fieldValue, float $x, float $y, float $pdfWidth, float $pdfHeight, array $coord): array
    {
        $fieldTypeLower = strtolower($fieldType);
        
        // Calculate relative position (as percentages)
        $xPercent = ($x / $pdfWidth) * 100;
        $yPercent = ($y / $pdfHeight) * 100;
        
        // Determine certificate zones
        $isInHeaderZone = $yPercent < 25;
        $isInMainZone = $yPercent >= 25 && $yPercent <= 75;
        $isInFooterZone = $yPercent > 75;
        $isInCenterHorizontal = $xPercent >= 20 && $xPercent <= 80;
        
        // Default alignment settings
        $alignment = 'L'; // Left align
        $adjustedX = $x;
        $adjustedY = $y;
        $cellWidth = 0; // Auto width
        
        // Apply contextual alignment rules
        if (strpos($fieldTypeLower, 'name') !== false) {
            // NAME FIELDS: Always center in main content area
            if ($isInMainZone) {
                $alignment = 'C';
                $cellWidth = min(300, $pdfWidth * 0.6); // 60% of page width or 300mm max
                $adjustedX = $x - ($cellWidth / 2); // Center the cell around the click point
            } else {
                // Names in header/footer areas - use moderate centering
                $alignment = 'C';
                $cellWidth = 200;
                $adjustedX = $x - 100;
            }
        } elseif (strpos($fieldTypeLower, 'course') !== false || strpos($fieldTypeLower, 'title') !== false || strpos($fieldTypeLower, 'subject') !== false) {
            // COURSE/TITLE FIELDS: Center if in main zone
            if ($isInMainZone) {
                $alignment = 'C';
                $cellWidth = min(350, $pdfWidth * 0.7);
                $adjustedX = $x - ($cellWidth / 2);
            }
        } elseif (strpos($fieldTypeLower, 'date') !== false) {
            // DATE FIELDS: Smart alignment based on position
            if ($xPercent < 50) {
                $alignment = 'L'; // Left side - left align
            } else {
                $alignment = 'R'; // Right side - right align
                $cellWidth = 120;
                $adjustedX = $x - 120;
            }
        } elseif (strpos($fieldTypeLower, 'signature') !== false || strpos($fieldTypeLower, 'instructor') !== false || strpos($fieldTypeLower, 'director') !== false) {
            // SIGNATURE FIELDS: Center alignment
            $alignment = 'C';
            $cellWidth = 180;
            $adjustedX = $x - 90;
        } elseif (strpos($fieldTypeLower, 'school') !== false || strpos($fieldTypeLower, 'organization') !== false) {
            // ORGANIZATION FIELDS: Center if in main area
            if ($isInMainZone && $isInCenterHorizontal) {
                $alignment = 'C';
                $cellWidth = 250;
                $adjustedX = $x - 125;
            }
        } else {
            // OTHER FIELDS: Smart alignment based on position
            if ($isInCenterHorizontal && $isInMainZone) {
                // Center area - center align
                $alignment = 'C';
                $cellWidth = 200;
                $adjustedX = $x - 100;
            } elseif ($xPercent > 75) {
                // Right area - right align
                $alignment = 'R';
                $cellWidth = 150;
                $adjustedX = $x - 150;
            }
            // Left area keeps default left alignment
        }
        
        // Ensure coordinates stay within bounds
        $adjustedX = max(5, min($adjustedX, $pdfWidth - $cellWidth - 5));
        $adjustedY = max(5, min($adjustedY, $pdfHeight - 15));
        
        return [
            'x' => $adjustedX,
            'y' => $adjustedY,
            'width' => $cellWidth,
            'align' => $alignment,
            'field_type' => $fieldTypeLower,
            'zone_info' => [
                'x_percent' => round($xPercent, 1),
                'y_percent' => round($yPercent, 1),
                'zone' => $isInHeaderZone ? 'header' : ($isInFooterZone ? 'footer' : 'main'),
                'horizontal_zone' => $isInCenterHorizontal ? 'center' : ($xPercent < 25 ? 'left' : 'right'),
                'applied_rule' => $fieldTypeLower . '_' . ($isInMainZone ? 'main' : ($isInHeaderZone ? 'header' : 'footer'))
            ]
        ];
    }

    /**
     * Set text color from hex color code
     */
    private function setTextColorFromHex(TCPDF $pdf, string $hexColor): void
    {
        // Remove # if present
        $hex = ltrim($hexColor, '#');
        
        // Convert hex to RGB
        if (strlen($hex) === 6) {
            $r = hexdec(substr($hex, 0, 2));
            $g = hexdec(substr($hex, 2, 2));
            $b = hexdec(substr($hex, 4, 2));
            $pdf->SetTextColor($r, $g, $b);
        } else {
            // Default to black if invalid hex
            $pdf->SetTextColor(0, 0, 0);
        }
    }

    /**
     * Add fields to image-based PDF (legacy method)
     */
    private function addFieldsToImageBasedPdf(TCPDF $pdf, array $fieldData, float $pdfWidth, float $pdfHeight): void
    {
        // Use the new overlay method
        $this->overlayFieldsOnOriginalTemplate($pdf, $fieldData, $pdfWidth, $pdfHeight, 0, 0);
    }

    /**
     * Add student-specific template elements
     */
    private function addStudentTemplateElements(TCPDF $pdf): void
    {
        // Student certificate styling
        $this->addCertificateBorder($pdf);
        
        // Add student-specific header
        $pdf->SetFont('helvetica', 'B', 24);
        $pdf->SetTextColor(0, 50, 100);
        $pdf->SetXY(20, 30);
        $pdf->Cell($pdf->getPageWidth() - 40, 15, 'STUDENT CERTIFICATE', 0, 1, 'C');
        
        // Add academic elements
        $pdf->SetFont('helvetica', '', 12);
        $pdf->SetTextColor(100, 100, 100);
        $pdf->SetXY(20, 50);
        $pdf->Cell($pdf->getPageWidth() - 40, 10, 'Academic Achievement Recognition', 0, 1, 'C');
    }

    /**
     * Add teacher-specific template elements
     */
    private function addTeacherTemplateElements(TCPDF $pdf): void
    {
        // Teacher certificate styling
        $this->addCertificateBorder($pdf);
        
        // Add teacher-specific header
        $pdf->SetFont('helvetica', 'B', 24);
        $pdf->SetTextColor(100, 0, 50);
        $pdf->SetXY(20, 30);
        $pdf->Cell($pdf->getPageWidth() - 40, 15, 'TEACHER CERTIFICATE', 0, 1, 'C');
        
        // Add professional elements
        $pdf->SetFont('helvetica', '', 12);
        $pdf->SetTextColor(100, 100, 100);
        $pdf->SetXY(20, 50);
        $pdf->Cell($pdf->getPageWidth() - 40, 10, 'Professional Development Recognition', 0, 1, 'C');
    }

    /**
     * Add generic template elements
     */
    private function addGenericTemplateElements(TCPDF $pdf): void
    {
        // Generic certificate styling
        $this->addCertificateBorder($pdf);
        
        // Add generic header
        $pdf->SetFont('helvetica', 'B', 24);
        $pdf->SetTextColor(0, 0, 0);
        $pdf->SetXY(20, 35);
        $pdf->Cell($pdf->getPageWidth() - 40, 15, 'CERTIFICATE', 0, 1, 'C');
        
        // Add subtitle
        $pdf->SetFont('helvetica', '', 14);
        $pdf->SetXY(20, 55);
        $pdf->Cell($pdf->getPageWidth() - 40, 10, 'This is to certify that', 0, 1, 'C');
    }

    /**
     * Add fields using provided coordinates with proper scaling
     */
    private function addFieldsWithCoordinates(TCPDF $pdf, array $fieldValues, array $fieldCoordinates): void
    {
        $pdfWidth = $pdf->getPageWidth();
        $pdfHeight = $pdf->getPageHeight();
        
        foreach ($fieldValues as $fieldName => $fieldValue) {
            if (empty($fieldValue)) continue;
            
            foreach ($fieldCoordinates as $coord) {
                if (strtolower($coord['type']) === strtolower($fieldName)) {
                    // Get coordinates - check both formats
                    $x = $coord['pdf_x'] ?? $coord['x'] ?? 0;
                    $y = $coord['pdf_y'] ?? $coord['y'] ?? 0;
                    
                    // Scale coordinates if preview dimensions are available
                    if (!empty($coord['preview_width']) && !empty($coord['preview_height'])) {
                        $scaleX = $pdfWidth / $coord['preview_width'];
                        $scaleY = $pdfHeight / $coord['preview_height'];
                        $x = $x * $scaleX;
                        $y = $y * $scaleY;
                    }
                    
                    $fontSize = $coord['font_size'] ?? 12;
                    $fontColor = $coord['font_color'] ?? '#000000';
                    
                    // Set font and color
                    $pdf->SetFont('helvetica', 'B', $fontSize);
                    $this->setTextColorFromHex($pdf, $fontColor);
                    
                    // Position and add text
                    $pdf->SetXY($x, $y);
                    $pdf->Cell(0, 10, $fieldValue, 0, 0, 'L');
                    break;
                }
            }
        }
    }

    /**
     * Create realistic certificate border
     */
    private function createRealisticCertificateBorder(TCPDF $pdf): void
    {
        $pageWidth = $pdf->getPageWidth();
        $pageHeight = $pdf->getPageHeight();
        
        // Outer decorative border
        $pdf->SetLineWidth(2);
        $pdf->SetDrawColor(0, 50, 100);
        $pdf->Rect(5, 5, $pageWidth - 10, $pageHeight - 10);
        
        // Inner elegant border
        $pdf->SetLineWidth(0.8);
        $pdf->SetDrawColor(150, 100, 50);
        $pdf->Rect(12, 12, $pageWidth - 24, $pageHeight - 24);
        
        // Decorative corner flourishes
        $this->addCornerFlourishes($pdf);
    }

    /**
     * Add corner flourishes for professional look
     */
    private function addCornerFlourishes(TCPDF $pdf): void
    {
        $pageWidth = $pdf->getPageWidth();
        $pageHeight = $pdf->getPageHeight();
        
        $pdf->SetLineWidth(0.5);
        $pdf->SetDrawColor(100, 100, 100);
        
        $cornerSize = 15;
        
        // Top-left corner
        $pdf->Line(20, 20, 20 + $cornerSize, 20);
        $pdf->Line(20, 20, 20, 20 + $cornerSize);
        
        // Top-right corner
        $pdf->Line($pageWidth - 20 - $cornerSize, 20, $pageWidth - 20, 20);
        $pdf->Line($pageWidth - 20, 20, $pageWidth - 20, 20 + $cornerSize);
        
        // Bottom-left corner
        $pdf->Line(20, $pageHeight - 20 - $cornerSize, 20, $pageHeight - 20);
        $pdf->Line(20, $pageHeight - 20, 20 + $cornerSize, $pageHeight - 20);
        
        // Bottom-right corner
        $pdf->Line($pageWidth - 20, $pageHeight - 20 - $cornerSize, $pageWidth - 20, $pageHeight - 20);
        $pdf->Line($pageWidth - 20 - $cornerSize, $pageHeight - 20, $pageWidth - 20, $pageHeight - 20);
    }

    /**
     * Add course completion specific elements
     */
    private function addCourseCompletionElements(TCPDF $pdf): void
    {
        $pdf->SetFont('helvetica', 'B', 26);
        $pdf->SetTextColor(0, 50, 100);
        $pdf->SetXY(20, 35);
        $pdf->Cell($pdf->getPageWidth() - 40, 20, 'CERTIFICATE OF COMPLETION', 0, 1, 'C');
        
        // Add decorative line
        $centerX = $pdf->getPageWidth() / 2;
        $pdf->SetLineWidth(1);
        $pdf->SetDrawColor(150, 100, 50);
        $pdf->Line($centerX - 80, 60, $centerX + 80, 60);
    }

    /**
     * Add achievement specific elements
     */
    private function addAchievementElements(TCPDF $pdf): void
    {
        $pdf->SetFont('helvetica', 'B', 26);
        $pdf->SetTextColor(100, 0, 50);
        $pdf->SetXY(20, 35);
        $pdf->Cell($pdf->getPageWidth() - 40, 20, 'CERTIFICATE OF ACHIEVEMENT', 0, 1, 'C');
        
        // Add star-like decorative elements
        $centerX = $pdf->getPageWidth() / 2;
        $pdf->SetLineWidth(0.8);
        $pdf->SetDrawColor(200, 150, 0);
        // Simple star representation with lines
        $pdf->Line($centerX - 10, 65, $centerX + 10, 65);
        $pdf->Line($centerX, 60, $centerX, 70);
    }

    /**
     * Add participation specific elements
     */
    private function addParticipationElements(TCPDF $pdf): void
    {
        $pdf->SetFont('helvetica', 'B', 24);
        $pdf->SetTextColor(0, 100, 0);
        $pdf->SetXY(20, 35);
        $pdf->Cell($pdf->getPageWidth() - 40, 20, 'CERTIFICATE OF PARTICIPATION', 0, 1, 'C');
    }

    /**
     * Add generic professional elements
     */
    private function addGenericProfessionalElements(TCPDF $pdf): void
    {
        $pdf->SetFont('helvetica', 'B', 28);
        $pdf->SetTextColor(0, 0, 0);
        $pdf->SetXY(20, 35);
        $pdf->Cell($pdf->getPageWidth() - 40, 20, 'CERTIFICATE', 0, 1, 'C');
        
        $pdf->SetFont('helvetica', '', 14);
        $pdf->SetXY(20, 65);
        $pdf->Cell($pdf->getPageWidth() - 40, 10, 'This is to certify that', 0, 1, 'C');
    }

    /**
     * Add fields with properly scaled coordinates
     */
    private function addFieldsWithScaledCoordinates(TCPDF $pdf, array $fieldValues, array $fieldCoordinates): void
    {
        $pdfWidth = $pdf->getPageWidth();
        $pdfHeight = $pdf->getPageHeight();
        
        foreach ($fieldValues as $fieldName => $fieldValue) {
            if (empty($fieldValue)) continue;
            
            // Find matching coordinate
            foreach ($fieldCoordinates as $coord) {
                if (strtolower($coord['type'] ?? '') === strtolower($fieldName)) {
                    // Get raw coordinates - check both formats
                    $x = $coord['pdf_x'] ?? $coord['x'] ?? 0;
                    $y = $coord['pdf_y'] ?? $coord['y'] ?? 0;
                    
                    // Scale coordinates from preview dimensions to PDF dimensions
                    if (!empty($coord['preview_width']) && !empty($coord['preview_height'])) {
                        $scaleX = $pdfWidth / $coord['preview_width'];
                        $scaleY = $pdfHeight / $coord['preview_height'];
                        $x = $x * $scaleX;
                        $y = $y * $scaleY;
                    }
                    
                    $fontSize = $coord['font_size'] ?? 14;
                    $fontColor = $coord['font_color'] ?? '#000000';
                    
                    // Set font and color
                    $pdf->SetFont('helvetica', 'B', $fontSize);
                    $this->setTextColorFromHex($pdf, $fontColor);
                    
                    // Position and add text
                    $pdf->SetXY($x, $y);
                    $pdf->Cell(0, 10, $fieldValue, 0, 0, 'L');
                    break;
                }
            }
        }
    }

    /**
     * Add fields with coordinates but fallback to intelligent positioning if needed
     */
    private function addFieldsWithCoordinatesFallback(TCPDF $pdf, array $fieldValues, array $fieldCoordinates, array $templateMetadata): void
    {
        $pageWidth = $pdf->getPageWidth();
        $pageHeight = $pdf->getPageHeight();
        
        foreach ($fieldValues as $fieldName => $fieldValue) {
            if (empty($fieldValue)) continue;
            
            // Find matching coordinate
            $coord = null;
            foreach ($fieldCoordinates as $c) {
                if (strtolower($c['type'] ?? '') === strtolower($fieldName)) {
                    $coord = $c;
                    break;
                }
            }
            
            if ($coord) {
                // Calculate scaled coordinates
                $x = $coord['pdf_x'] ?? $coord['x'] ?? 0;
                $y = $coord['pdf_y'] ?? $coord['y'] ?? 0;
                
                if (!empty($coord['preview_width']) && !empty($coord['preview_height'])) {
                    $scaleX = $pageWidth / $coord['preview_width'];
                    $scaleY = $pageHeight / $coord['preview_height'];
                    $x = $x * $scaleX;
                    $y = $y * $scaleY;
                }
                
                // Check if coordinates are reasonable for our recreated template
                if ($x > 0 && $x < $pageWidth && $y > 60 && $y < $pageHeight - 40) {
                    // Coordinates seem reasonable, use them
                    $fontSize = max($coord['font_size'] ?? 14, 14); // Minimum 14pt
                    $fontColor = $coord['font_color'] ?? '#000000';
                    
                    $pdf->SetFont('helvetica', 'B', $fontSize);
                    $this->setTextColorFromHex($pdf, $fontColor);
                    
                    // Add text with enhanced visibility
                    $pdf->SetXY($x, $y);
                    $pdf->Cell(0, 10, $fieldValue, 0, 0, 'L');
                } else {
                    // Coordinates don't make sense for our template, use intelligent positioning
                    $this->addFieldWithIntelligentPosition($pdf, $fieldName, $fieldValue, $pageWidth, $pageHeight);
                }
            } else {
                // No coordinates found, use intelligent positioning
                $this->addFieldWithIntelligentPosition($pdf, $fieldName, $fieldValue, $pageWidth, $pageHeight);
            }
        }
    }
    
    /**
     * Add a single field with intelligent positioning
     */
    private function addFieldWithIntelligentPosition(TCPDF $pdf, string $fieldName, string $fieldValue, float $pageWidth, float $pageHeight): void
    {
        // Smart positions based on field name and certificate layout
        $positions = [
            'name' => ['x' => $pageWidth * 0.5, 'y' => 90, 'size' => 20, 'align' => 'C', 'color' => '#000080'],
            'course' => ['x' => $pageWidth * 0.5, 'y' => 110, 'size' => 16, 'align' => 'C', 'color' => '#800000'],
            'date' => ['x' => $pageWidth * 0.2, 'y' => $pageHeight - 50, 'size' => 12, 'align' => 'L', 'color' => '#000000'],
            'instructor' => ['x' => $pageWidth * 0.8, 'y' => $pageHeight - 50, 'size' => 12, 'align' => 'R', 'color' => '#000000'],
        ];
        
        $fieldKey = strtolower($fieldName);
        $pos = $positions[$fieldKey] ?? ['x' => $pageWidth * 0.5, 'y' => 100, 'size' => 16, 'align' => 'C', 'color' => '#000000'];
        
        $pdf->SetFont('helvetica', 'B', $pos['size']);
        $this->setTextColorFromHex($pdf, $pos['color']);
        
        if ($pos['align'] === 'C') {
            $pdf->SetXY($pos['x'] - 60, $pos['y']);
            $pdf->Cell(120, 10, $fieldValue, 0, 0, 'C');
        } elseif ($pos['align'] === 'R') {
            $pdf->SetXY($pos['x'] - 120, $pos['y']);
            $pdf->Cell(120, 10, $fieldValue, 0, 0, 'R');
        } else {
            $pdf->SetXY($pos['x'], $pos['y']);
            $pdf->Cell(120, 10, $fieldValue, 0, 0, 'L');
        }
    }

    /**
     * Add fields with advanced positioning logic
     */
    private function addFieldsWithAdvancedPositioning(TCPDF $pdf, array $fieldValues, array $templateMetadata): void
    {
        $pageWidth = $pdf->getPageWidth();
        $pageHeight = $pdf->getPageHeight();
        
        // Define smart positions based on certificate layout
        $positions = [
            'name' => ['x' => $pageWidth * 0.5, 'y' => $pageHeight * 0.45, 'size' => 20, 'align' => 'C', 'color' => '#000080'],
            'course' => ['x' => $pageWidth * 0.5, 'y' => $pageHeight * 0.55, 'size' => 16, 'align' => 'C', 'color' => '#800000'],
            'date' => ['x' => $pageWidth * 0.2, 'y' => $pageHeight * 0.8, 'size' => 12, 'align' => 'L', 'color' => '#000000'],
            'instructor' => ['x' => $pageWidth * 0.8, 'y' => $pageHeight * 0.8, 'size' => 12, 'align' => 'R', 'color' => '#000000'],
            'duration' => ['x' => $pageWidth * 0.2, 'y' => $pageHeight * 0.75, 'size' => 11, 'align' => 'L', 'color' => '#666666'],
            'grade' => ['x' => $pageWidth * 0.8, 'y' => $pageHeight * 0.75, 'size' => 14, 'align' => 'R', 'color' => '#008000'],
        ];
        
        foreach ($fieldValues as $fieldName => $fieldValue) {
            if (empty($fieldValue)) continue;
            
            $pos = $positions[$fieldName] ?? ['x' => $pageWidth * 0.5, 'y' => $pageHeight * 0.65, 'size' => 12, 'align' => 'C', 'color' => '#000000'];
            
            $pdf->SetFont('helvetica', 'B', $pos['size']);
            $this->setTextColorFromHex($pdf, $pos['color']);
            
            if ($pos['align'] === 'C') {
                $pdf->SetXY($pos['x'] - 60, $pos['y']);
                $pdf->Cell(120, 10, $fieldValue, 0, 0, 'C');
            } elseif ($pos['align'] === 'R') {
                $pdf->SetXY($pos['x'] - 120, $pos['y']);
                $pdf->Cell(120, 10, $fieldValue, 0, 0, 'R');
            } else {
                $pdf->SetXY($pos['x'], $pos['y']);
                $pdf->Cell(120, 10, $fieldValue, 0, 0, 'L');
            }
        }
    }
}