<?php

namespace App\Service;

use Exception;

/**
 * Enhanced PDF error logging service with rotation and retention
 */
class PdfErrorLogger
{
    private string $logDirectory;
    private string $logPrefix;
    private int $maxLogSize;
    private int $retentionDays;
    private string $currentLogFile;

    public function __construct(
        string $logDirectory = 'logs',
        string $logPrefix = 'pdf_errors',
        int $maxLogSize = 10485760, // 10MB
        int $retentionDays = 30
    ) {
        $this->logDirectory = $logDirectory;
        $this->logPrefix = $logPrefix;
        $this->maxLogSize = $maxLogSize;
        $this->retentionDays = $retentionDays;
        $this->currentLogFile = $this->getCurrentLogFile();
        
        $this->ensureLogDirectoryExists();
        $this->cleanupOldLogs();
    }

    /**
     * Log PDF processing error with structured data
     */
    public function logError(
        string $templateId,
        string $templatePath,
        Exception $exception,
        array $context = []
    ): void {
        $logEntry = $this->createLogEntry($templateId, $templatePath, $exception, $context);
        $this->writeLogEntry($logEntry);
        
        // Check if rotation is needed
        $this->rotateLogIfNeeded();
    }

    /**
     * Log PDF processing success for monitoring
     */
    public function logSuccess(
        string $templateId,
        string $templatePath,
        string $processingMethod,
        float $processingTime,
        array $context = []
    ): void {
        $logEntry = [
            'timestamp' => $this->getCurrentTimestamp(),
            'level' => 'INFO',
            'event_type' => 'pdf_processing_success',
            'template_id' => $templateId,
            'template_path' => $templatePath,
            'processing_method' => $processingMethod,
            'processing_time_ms' => round($processingTime * 1000, 2),
            'context' => $context,
            'session_id' => session_id() ?: 'no_session',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'ip_address' => $this->getClientIpAddress(),
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true)
        ];

        $this->writeLogEntry($logEntry);
    }

    /**
     * Log template validation results
     */
    public function logValidation(
        string $templateId,
        string $templatePath,
        bool $isValid,
        array $validationResults,
        array $context = []
    ): void {
        $logEntry = [
            'timestamp' => $this->getCurrentTimestamp(),
            'level' => $isValid ? 'INFO' : 'WARNING',
            'event_type' => 'template_validation',
            'template_id' => $templateId,
            'template_path' => $templatePath,
            'is_valid' => $isValid,
            'validation_results' => $validationResults,
            'context' => $context,
            'session_id' => session_id() ?: 'no_session',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'ip_address' => $this->getClientIpAddress()
        ];

        $this->writeLogEntry($logEntry);
    }

    /**
     * Get recent log entries
     */
    public function getRecentEntries(int $limit = 100, string $level = null): array
    {
        $entries = [];
        $logFiles = $this->getLogFiles();
        
        // Sort by modification time (newest first)
        usort($logFiles, function($a, $b) {
            return filemtime($b) - filemtime($a);
        });

        foreach ($logFiles as $logFile) {
            if (!file_exists($logFile)) {
                continue;
            }

            $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $lines = array_reverse($lines); // Most recent first

            foreach ($lines as $line) {
                $entry = json_decode($line, true);
                
                if ($entry && ($level === null || $entry['level'] === $level)) {
                    $entries[] = $entry;
                    
                    if (count($entries) >= $limit) {
                        break 2; // Break out of both loops
                    }
                }
            }
        }

        return $entries;
    }

    /**
     * Get error statistics for a time period
     */
    public function getErrorStatistics(int $days = 7): array
    {
        $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        $entries = $this->getRecentEntries(10000); // Get a large number to analyze
        
        $stats = [
            'total_errors' => 0,
            'total_successes' => 0,
            'error_rate' => 0,
            'by_template' => [],
            'by_error_type' => [],
            'by_processing_method' => [],
            'avg_processing_time' => 0,
            'time_period' => $days . ' days'
        ];

        $processingTimes = [];

        foreach ($entries as $entry) {
            if ($entry['timestamp'] < $cutoffDate) {
                continue;
            }

            if ($entry['level'] === 'ERROR') {
                $stats['total_errors']++;
                
                // Count by template
                $templateId = $entry['template_id'];
                if (!isset($stats['by_template'][$templateId])) {
                    $stats['by_template'][$templateId] = ['errors' => 0, 'successes' => 0];
                }
                $stats['by_template'][$templateId]['errors']++;
                
                // Count by error type
                $errorType = $this->categorizeError($entry);
                $stats['by_error_type'][$errorType] = ($stats['by_error_type'][$errorType] ?? 0) + 1;
                
            } elseif ($entry['event_type'] === 'pdf_processing_success') {
                $stats['total_successes']++;
                
                // Count by template
                $templateId = $entry['template_id'];
                if (!isset($stats['by_template'][$templateId])) {
                    $stats['by_template'][$templateId] = ['errors' => 0, 'successes' => 0];
                }
                $stats['by_template'][$templateId]['successes']++;
                
                // Count by processing method
                $method = $entry['processing_method'] ?? 'unknown';
                $stats['by_processing_method'][$method] = ($stats['by_processing_method'][$method] ?? 0) + 1;
                
                // Collect processing times
                if (isset($entry['processing_time_ms'])) {
                    $processingTimes[] = $entry['processing_time_ms'];
                }
            }
        }

        // Calculate error rate
        $total = $stats['total_errors'] + $stats['total_successes'];
        $stats['error_rate'] = $total > 0 ? round(($stats['total_errors'] / $total) * 100, 2) : 0;
        
        // Calculate average processing time
        $stats['avg_processing_time'] = count($processingTimes) > 0 ? 
            round(array_sum($processingTimes) / count($processingTimes), 2) : 0;

        return $stats;
    }

    /**
     * Create structured log entry
     */
    private function createLogEntry(
        string $templateId,
        string $templatePath,
        Exception $exception,
        array $context
    ): array {
        return [
            'timestamp' => $this->getCurrentTimestamp(),
            'level' => 'ERROR',
            'event_type' => 'pdf_processing_error',
            'template_id' => $templateId,
            'template_path' => $templatePath,
            'error_class' => get_class($exception),
            'error_message' => $exception->getMessage(),
            'error_code' => $exception->getCode(),
            'stack_trace' => $exception->getTraceAsString(),
            'context' => $context,
            'session_id' => session_id() ?: 'no_session',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'ip_address' => $this->getClientIpAddress(),
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true),
            'php_version' => PHP_VERSION,
            'server_name' => $_SERVER['SERVER_NAME'] ?? 'Unknown'
        ];
    }

    /**
     * Write log entry to file
     */
    private function writeLogEntry(array $logEntry): void
    {
        $logLine = json_encode($logEntry, JSON_UNESCAPED_SLASHES) . PHP_EOL;
        
        // Ensure current log file exists
        if (!file_exists($this->currentLogFile)) {
            touch($this->currentLogFile);
            chmod($this->currentLogFile, 0644);
        }
        
        file_put_contents($this->currentLogFile, $logLine, FILE_APPEND | LOCK_EX);
    }

    /**
     * Get current log file path
     */
    private function getCurrentLogFile(): string
    {
        $date = date('Y-m-d');
        return $this->logDirectory . DIRECTORY_SEPARATOR . $this->logPrefix . '_' . $date . '.log';
    }

    /**
     * Get all log files
     */
    private function getLogFiles(): array
    {
        $pattern = $this->logDirectory . DIRECTORY_SEPARATOR . $this->logPrefix . '_*.log';
        return glob($pattern) ?: [];
    }

    /**
     * Rotate log file if it exceeds maximum size
     */
    private function rotateLogIfNeeded(): void
    {
        if (!file_exists($this->currentLogFile)) {
            return;
        }

        if (filesize($this->currentLogFile) > $this->maxLogSize) {
            $timestamp = date('Y-m-d_H-i-s');
            $rotatedFile = str_replace('.log', '_' . $timestamp . '.log', $this->currentLogFile);
            
            rename($this->currentLogFile, $rotatedFile);
            
            // Update current log file reference
            $this->currentLogFile = $this->getCurrentLogFile();
        }
    }

    /**
     * Clean up old log files based on retention policy
     */
    private function cleanupOldLogs(): void
    {
        $logFiles = $this->getLogFiles();
        $cutoffTime = time() - ($this->retentionDays * 24 * 60 * 60);

        foreach ($logFiles as $logFile) {
            if (filemtime($logFile) < $cutoffTime) {
                unlink($logFile);
            }
        }
    }

    /**
     * Ensure log directory exists
     */
    private function ensureLogDirectoryExists(): void
    {
        if (!is_dir($this->logDirectory)) {
            mkdir($this->logDirectory, 0755, true);
        }
    }

    /**
     * Get current timestamp in ISO format
     */
    private function getCurrentTimestamp(): string
    {
        return date('Y-m-d H:i:s');
    }

    /**
     * Get client IP address
     */
    private function getClientIpAddress(): string
    {
        $ipKeys = ['HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'REMOTE_ADDR'];
        
        foreach ($ipKeys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                // Handle comma-separated IPs (from proxies)
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                return $ip;
            }
        }
        
        return 'Unknown';
    }

    /**
     * Categorize error for statistics
     */
    private function categorizeError(array $entry): string
    {
        $errorClass = $entry['error_class'] ?? '';
        $errorMessage = $entry['error_message'] ?? '';

        if (strpos($errorClass, 'CrossReference') !== false || 
            strpos($errorMessage, 'CrossReference') !== false) {
            return 'Compression Error';
        }
        
        if (strpos($errorClass, 'Parser') !== false || 
            strpos($errorMessage, 'Parser') !== false) {
            return 'Parsing Error';
        }
        
        if (strpos($errorMessage, 'not found') !== false || 
            strpos($errorMessage, 'No such file') !== false) {
            return 'File Not Found';
        }
        
        if (strpos($errorMessage, 'Fallback') !== false) {
            return 'Fallback Error';
        }
        
        if (strpos($errorMessage, 'memory') !== false) {
            return 'Memory Error';
        }
        
        if (strpos($errorMessage, 'timeout') !== false) {
            return 'Timeout Error';
        }
        
        return 'Other';
    }
}