<?php

namespace App\Service;

use Exception;

/**
 * Service for managing PDF templates and their compatibility status
 */
class TemplateManagementService
{
    private TemplateValidatorInterface $validator;
    private ErrorHandlerInterface $errorHandler;
    private string $templatesDirectory;

    public function __construct(
        TemplateValidatorInterface $validator,
        ErrorHandlerInterface $errorHandler,
        string $templatesDirectory = 'templates'
    ) {
        $this->validator = $validator;
        $this->errorHandler = $errorHandler;
        $this->templatesDirectory = $templatesDirectory;
    }

    /**
     * Batch validate all templates in the templates directory
     */
    public function batchValidateTemplates(): array
    {
        $results = [
            'total_templates' => 0,
            'valid_templates' => 0,
            'invalid_templates' => 0,
            'fpdi_compatible' => 0,
            'fpdi_incompatible' => 0,
            'templates' => [],
            'summary' => []
        ];

        if (!is_dir($this->templatesDirectory)) {
            throw new Exception("Templates directory not found: {$this->templatesDirectory}");
        }

        $templateFiles = $this->getTemplateFiles();
        $results['total_templates'] = count($templateFiles);

        foreach ($templateFiles as $templateFile) {
            $templatePath = $this->templatesDirectory . DIRECTORY_SEPARATOR . $templateFile;
            $templateId = $this->extractTemplateId($templateFile);
            
            try {
                $validationResult = $this->validator->validateTemplate($templatePath);
                
                $templateInfo = [
                    'id' => $templateId,
                    'filename' => $templateFile,
                    'path' => $templatePath,
                    'file_size' => file_exists($templatePath) ? filesize($templatePath) : 0,
                    'is_valid' => $validationResult->isValid,
                    'fpdi_compatible' => $validationResult->fpdiCompatible,
                    'issues' => $validationResult->issues,
                    'recommendations' => $validationResult->recommendations,
                    'validation_timestamp' => date('Y-m-d H:i:s')
                ];

                // Update counters
                if ($validationResult->isValid) {
                    $results['valid_templates']++;
                    
                    if ($validationResult->fpdiCompatible) {
                        $results['fpdi_compatible']++;
                        $templateInfo['status'] = 'compatible';
                    } else {
                        $results['fpdi_incompatible']++;
                        $templateInfo['status'] = 'incompatible_but_usable';
                    }
                } else {
                    $results['invalid_templates']++;
                    $templateInfo['status'] = 'invalid';
                }

                $results['templates'][$templateId] = $templateInfo;

            } catch (\Exception $e) {
                $results['invalid_templates']++;
                $results['templates'][$templateId] = [
                    'id' => $templateId,
                    'filename' => $templateFile,
                    'path' => $templatePath,
                    'file_size' => 0,
                    'is_valid' => false,
                    'fpdi_compatible' => false,
                    'status' => 'error',
                    'error' => $e->getMessage(),
                    'validation_timestamp' => date('Y-m-d H:i:s')
                ];

                // Log validation error
                $this->errorHandler->logPdfError(
                    $templateId,
                    "Template validation error: " . $e->getMessage(),
                    [
                        'templatePath' => $templatePath,
                        'exception' => $e,
                        'operation' => 'batch_validation'
                    ]
                );
            }
        }

        // Generate summary
        $results['summary'] = $this->generateValidationSummary($results);

        return $results;
    }

    /**
     * Get compatibility status for all templates
     */
    public function getTemplateCompatibilityStatus(): array
    {
        $templateFiles = $this->getTemplateFiles();
        $status = [];

        foreach ($templateFiles as $templateFile) {
            $templatePath = $this->templatesDirectory . DIRECTORY_SEPARATOR . $templateFile;
            $templateId = $this->extractTemplateId($templateFile);
            
            $status[$templateId] = [
                'id' => $templateId,
                'filename' => $templateFile,
                'path' => $templatePath,
                'file_size' => file_exists($templatePath) ? filesize($templatePath) : 0,
                'last_modified' => file_exists($templatePath) ? date('Y-m-d H:i:s', filemtime($templatePath)) : null,
                'fpdi_compatible' => $this->validator->checkPdfCompatibility($templatePath),
                'recent_errors' => $this->errorHandler->getRecentErrors($templateId, 5)
            ];
        }

        return $status;
    }

    /**
     * Get problematic templates that need attention
     */
    public function getProblematicTemplates(): array
    {
        $allTemplates = $this->getTemplateCompatibilityStatus();
        $problematic = [];

        foreach ($allTemplates as $templateId => $templateInfo) {
            $isProblematic = false;
            $issues = [];

            // Check if file doesn't exist
            if (!file_exists($templateInfo['path'])) {
                $isProblematic = true;
                $issues[] = 'File not found';
            }

            // Check if not FPDI compatible
            if (!$templateInfo['fpdi_compatible']) {
                $isProblematic = true;
                $issues[] = 'Not FPDI compatible - requires fallback generation';
            }

            // Check if has recent errors
            if (count($templateInfo['recent_errors']) > 0) {
                $isProblematic = true;
                $issues[] = count($templateInfo['recent_errors']) . ' recent errors';
            }

            if ($isProblematic) {
                $templateInfo['issues'] = $issues;
                $problematic[$templateId] = $templateInfo;
            }
        }

        return $problematic;
    }

    /**
     * Generate template replacement recommendations
     */
    public function generateReplacementRecommendations(): array
    {
        $problematic = $this->getProblematicTemplates();
        $compatible = array_filter(
            $this->getTemplateCompatibilityStatus(),
            function($template) {
                return $template['fpdi_compatible'] && file_exists($template['path']);
            }
        );

        $recommendations = [];

        foreach ($problematic as $templateId => $templateInfo) {
            $recommendation = [
                'template_id' => $templateId,
                'issues' => $templateInfo['issues'],
                'priority' => $this->calculateReplacementPriority($templateInfo),
                'suggested_actions' => []
            ];

            // Generate specific recommendations based on issues
            if (in_array('File not found', $templateInfo['issues'])) {
                $recommendation['suggested_actions'][] = 'Re-upload the template file';
                $recommendation['priority'] = 'high';
            }

            if (in_array('Not FPDI compatible - requires fallback generation', $templateInfo['issues'])) {
                $recommendation['suggested_actions'][] = 'Convert PDF to FPDI-compatible format';
                $recommendation['suggested_actions'][] = 'Test fallback generation quality';
                
                // Suggest similar compatible templates
                $similarTemplates = $this->findSimilarCompatibleTemplates($templateInfo, $compatible);
                if (!empty($similarTemplates)) {
                    $recommendation['suggested_alternatives'] = $similarTemplates;
                }
            }

            if (strpos(implode(' ', $templateInfo['issues']), 'recent errors') !== false) {
                $recommendation['suggested_actions'][] = 'Review recent error logs';
                $recommendation['suggested_actions'][] = 'Test template with sample data';
            }

            $recommendations[] = $recommendation;
        }

        // Sort by priority
        usort($recommendations, function($a, $b) {
            $priorities = ['high' => 3, 'medium' => 2, 'low' => 1];
            return ($priorities[$b['priority']] ?? 0) - ($priorities[$a['priority']] ?? 0);
        });

        return $recommendations;
    }

    /**
     * Create template usage report
     */
    public function generateUsageReport(int $days = 30): array
    {
        $stats = $this->errorHandler->getEnhancedErrorStats($days);
        $templates = $this->getTemplateCompatibilityStatus();

        $report = [
            'period' => $days . ' days',
            'total_templates' => count($templates),
            'error_rate' => $stats['error_rate'] ?? 0,
            'most_used_templates' => [],
            'most_problematic_templates' => [],
            'processing_methods' => $stats['by_processing_method'] ?? [],
            'recommendations' => []
        ];

        // Analyze template usage from stats
        if (isset($stats['by_template'])) {
            foreach ($stats['by_template'] as $templateId => $templateStats) {
                $total = $templateStats['errors'] + $templateStats['successes'];
                $errorRate = $total > 0 ? ($templateStats['errors'] / $total) * 100 : 0;

                $report['most_used_templates'][] = [
                    'template_id' => $templateId,
                    'total_usage' => $total,
                    'successes' => $templateStats['successes'],
                    'errors' => $templateStats['errors'],
                    'error_rate' => round($errorRate, 2)
                ];

                if ($errorRate > 10) { // Templates with >10% error rate
                    $report['most_problematic_templates'][] = [
                        'template_id' => $templateId,
                        'error_rate' => round($errorRate, 2),
                        'total_errors' => $templateStats['errors'],
                        'total_usage' => $total
                    ];
                }
            }
        }

        // Sort by usage
        usort($report['most_used_templates'], function($a, $b) {
            return $b['total_usage'] - $a['total_usage'];
        });

        // Sort problematic by error rate
        usort($report['most_problematic_templates'], function($a, $b) {
            return $b['error_rate'] - $a['error_rate'];
        });

        // Generate recommendations
        $report['recommendations'] = $this->generateUsageRecommendations($report);

        return $report;
    }

    /**
     * Get template files from directory
     */
    private function getTemplateFiles(): array
    {
        if (!is_dir($this->templatesDirectory)) {
            return [];
        }

        $files = scandir($this->templatesDirectory);
        return array_filter($files, function($file) {
            return pathinfo($file, PATHINFO_EXTENSION) === 'pdf';
        });
    }

    /**
     * Extract template ID from filename
     */
    private function extractTemplateId(string $filename): string
    {
        // Remove .pdf extension and use filename as ID
        return pathinfo($filename, PATHINFO_FILENAME);
    }

    /**
     * Generate validation summary
     */
    private function generateValidationSummary(array $results): array
    {
        $summary = [];

        if ($results['total_templates'] === 0) {
            $summary[] = 'No templates found in directory';
            return $summary;
        }

        $validPercentage = round(($results['valid_templates'] / $results['total_templates']) * 100, 1);
        $compatiblePercentage = round(($results['fpdi_compatible'] / $results['total_templates']) * 100, 1);

        $summary[] = "{$results['total_templates']} templates processed";
        $summary[] = "{$results['valid_templates']} valid templates ({$validPercentage}%)";
        $summary[] = "{$results['fpdi_compatible']} FPDI compatible templates ({$compatiblePercentage}%)";

        if ($results['invalid_templates'] > 0) {
            $summary[] = "{$results['invalid_templates']} invalid templates require attention";
        }

        if ($results['fpdi_incompatible'] > 0) {
            $summary[] = "{$results['fpdi_incompatible']} templates will use fallback generation";
        }

        return $summary;
    }

    /**
     * Calculate replacement priority for problematic templates
     */
    private function calculateReplacementPriority(array $templateInfo): string
    {
        if (in_array('File not found', $templateInfo['issues'])) {
            return 'high';
        }

        if (count($templateInfo['recent_errors']) > 3) {
            return 'high';
        }

        if (count($templateInfo['recent_errors']) > 0) {
            return 'medium';
        }

        return 'low';
    }

    /**
     * Find similar compatible templates
     */
    private function findSimilarCompatibleTemplates(array $problematicTemplate, array $compatibleTemplates): array
    {
        $similar = [];
        $problematicSize = $problematicTemplate['file_size'];

        foreach ($compatibleTemplates as $templateId => $template) {
            // Simple similarity based on file size (within 20% difference)
            $sizeDifference = abs($template['file_size'] - $problematicSize) / $problematicSize;
            
            if ($sizeDifference <= 0.2) {
                $similar[] = [
                    'template_id' => $templateId,
                    'filename' => $template['filename'],
                    'size_similarity' => round((1 - $sizeDifference) * 100, 1) . '%'
                ];
            }
        }

        return array_slice($similar, 0, 3); // Return top 3 similar templates
    }

    /**
     * Generate usage-based recommendations
     */
    private function generateUsageRecommendations(array $report): array
    {
        $recommendations = [];

        if ($report['error_rate'] > 15) {
            $recommendations[] = 'High overall error rate detected - review template quality and compatibility';
        }

        if (count($report['most_problematic_templates']) > 0) {
            $recommendations[] = 'Focus on fixing the most problematic templates to improve overall reliability';
        }

        $fallbackUsage = $report['processing_methods']['fallback'] ?? 0;
        $totalProcessing = array_sum($report['processing_methods']);
        
        if ($totalProcessing > 0 && ($fallbackUsage / $totalProcessing) > 0.3) {
            $recommendations[] = 'High fallback usage detected - consider converting templates to FPDI-compatible format';
        }

        if (empty($recommendations)) {
            $recommendations[] = 'Template performance looks good - continue monitoring';
        }

        return $recommendations;
    }
}