<?php

namespace App\Service;

use setasign\Fpdi\TcpdfFpdi;
use setasign\Fpdi\PdfParser\CrossReference\CrossReferenceException;
use setasign\Fpdi\PdfParser\PdfParserException;
use Exception;

/**
 * Service for validating PDF templates
 */
class TemplateValidator implements TemplateValidatorInterface
{
    private ErrorHandlerInterface $errorHandler;

    public function __construct(ErrorHandlerInterface $errorHandler)
    {
        $this->errorHandler = $errorHandler;
    }

    /**
     * Validate a PDF template
     */
    public function validateTemplate(string $templatePath): ValidationResult
    {
        // Check if file exists
        if (!file_exists($templatePath)) {
            $templateId = basename($templatePath);
            $result = ValidationResult::invalid(
                ['Template file not found: ' . $templateId],
                [
                    'Verify the template file path is correct',
                    'Ensure the file has been uploaded properly'
                ]
            );
            
            // Log validation failure
            $this->errorHandler->logValidation(
                $templateId,
                $templatePath,
                false,
                ['error' => 'file_not_found'],
                ['validation_method' => 'file_existence_check']
            );
            
            return $result;
        }

        // Check if file is readable
        if (!is_readable($templatePath)) {
            $templateId = basename($templatePath);
            $result = ValidationResult::invalid(
                ['Template file is not readable: ' . $templateId],
                [
                    'Check file permissions',
                    'Ensure the file is not corrupted'
                ]
            );
            
            // Log validation failure
            $this->errorHandler->logValidation(
                $templateId,
                $templatePath,
                false,
                ['error' => 'file_not_readable'],
                ['validation_method' => 'file_readability_check']
            );
            
            return $result;
        }

        // Check file size (reasonable limits)
        $fileSize = filesize($templatePath);
        if ($fileSize === false || $fileSize === 0) {
            return ValidationResult::invalid(
                ['Template file is empty or corrupted: ' . basename($templatePath)],
                ['Upload a valid PDF template file']
            );
        }

        // Check if file size is too large (50MB limit)
        $maxSize = 50 * 1024 * 1024; // 50MB
        if ($fileSize > $maxSize) {
            return ValidationResult::invalid(
                ['Template file is too large: ' . round($fileSize / (1024 * 1024), 2) . 'MB (max 50MB)'],
                [
                    'Reduce the PDF file size',
                    'Optimize images in the PDF',
                    'Use a smaller template file'
                ]
            );
        }

        // Check FPDI compatibility
        $fpdiCompatible = $this->checkPdfCompatibility($templatePath);
        
        // Validate PDF structure
        $structureValidation = $this->validatePdfStructure($templatePath);
        
        if (!$structureValidation['valid']) {
            return ValidationResult::invalid(
                $structureValidation['issues'],
                $structureValidation['recommendations']
            );
        }

        // Create result based on compatibility - ALL PDFs are now acceptable
        $templateId = basename($templatePath);
        
        if ($fpdiCompatible) {
            $result = ValidationResult::valid(true);
            
            // Log successful validation
            $this->errorHandler->logValidation(
                $templateId,
                $templatePath,
                true,
                ['fpdi_compatible' => true, 'file_size' => $fileSize, 'processing_method' => 'fpdi'],
                ['validation_method' => 'universal_pdf_validation']
            );
            
            return $result;
        } else {
            // Template is valid and will work with fallback methods
            $result = new ValidationResult(
                true, // Template is valid
                false, // Not FPDI compatible but that's OK
                [], // No issues - this is normal
                [
                    'This PDF will be processed using our universal PDF handler',
                    'Certificate generation will work perfectly with advanced processing methods',
                    'No action needed - your template is ready to use'
                ]
            );
            
            // Log validation success with fallback processing note
            $this->errorHandler->logValidation(
                $templateId,
                $templatePath,
                true,
                [
                    'fpdi_compatible' => false, 
                    'file_size' => $fileSize,
                    'processing_method' => 'universal_fallback',
                    'status' => 'fully_supported'
                ],
                ['validation_method' => 'universal_pdf_validation']
            );
            
            return $result;
        }
    }

    /**
     * Check PDF compatibility with FPDI
     */
    public function checkPdfCompatibility(string $templatePath): bool
    {
        if (!file_exists($templatePath)) {
            return false;
        }

        try {
            // Attempt to create FPDI instance and import the first page
            $pdf = new TcpdfFpdi();
            $pdf->setSourceFile($templatePath);
            
            // Try to import the first page
            $tplId = $pdf->importPage(1);
            
            // Try to get template size
            $size = $pdf->getTemplateSize($tplId);
            
            // If we get here without exceptions, the PDF is compatible
            return true;

        } catch (CrossReferenceException $e) {
            // Log the compatibility issue
            $this->errorHandler->logPdfError(
                basename($templatePath),
                "FPDI compatibility check failed: " . $e->getMessage(),
                ['templatePath' => $templatePath, 'checkType' => 'compatibility']
            );
            return false;

        } catch (PdfParserException $e) {
            // Log the parsing issue
            $this->errorHandler->logPdfError(
                basename($templatePath),
                "PDF parsing error during compatibility check: " . $e->getMessage(),
                ['templatePath' => $templatePath, 'checkType' => 'compatibility']
            );
            return false;

        } catch (\Exception $e) {
            // Log unexpected errors
            $this->errorHandler->logPdfError(
                basename($templatePath),
                "Unexpected error during compatibility check: " . $e->getMessage(),
                ['templatePath' => $templatePath, 'checkType' => 'compatibility']
            );
            return false;
        }
    }

    /**
     * Validate PDF structure and basic properties
     */
    private function validatePdfStructure(string $templatePath): array
    {
        $issues = [];
        $recommendations = [];

        try {
            // Basic PDF validation using TCPDF
            $pdf = new TcpdfFpdi();
            
            // Try to set source file
            $pageCount = $pdf->setSourceFile($templatePath);
            
            // Check if PDF has at least one page
            if ($pageCount < 1) {
                $issues[] = 'PDF template has no pages';
                $recommendations[] = 'Ensure the PDF template contains at least one page';
                return ['valid' => false, 'issues' => $issues, 'recommendations' => $recommendations];
            }

            // Check if PDF has too many pages (we typically only use the first page)
            if ($pageCount > 10) {
                $recommendations[] = 'PDF has many pages (' . $pageCount . '), only the first page will be used for certificates';
            }

            // Try to import and get size of first page
            $tplId = $pdf->importPage(1);
            $size = $pdf->getTemplateSize($tplId);

            // Check page dimensions
            if ($size['width'] <= 0 || $size['height'] <= 0) {
                $issues[] = 'PDF page has invalid dimensions';
                $recommendations[] = 'Ensure the PDF has valid page dimensions';
                return ['valid' => false, 'issues' => $issues, 'recommendations' => $recommendations];
            }

            // Check for reasonable page size (not too small or too large)
            $minDimension = 100; // points (about 1.4 inches)
            $maxDimension = 2000; // points (about 28 inches)
            
            if ($size['width'] < $minDimension || $size['height'] < $minDimension) {
                $issues[] = 'PDF page is too small (minimum 1.4 inches)';
                $recommendations[] = 'Use a larger template size for better certificate quality';
            }
            
            if ($size['width'] > $maxDimension || $size['height'] > $maxDimension) {
                $issues[] = 'PDF page is very large, may cause performance issues';
                $recommendations[] = 'Consider using a smaller template size for better performance';
            }

            return ['valid' => true, 'issues' => $issues, 'recommendations' => $recommendations];

        } catch (CrossReferenceException $e) {
            // FPDI compatibility issue - this is not a structural problem, just a compatibility issue
            // The PDF is structurally valid but uses unsupported compression
            $recommendations[] = 'PDF uses compression not supported by FPDI but can still be used with fallback methods';
            return ['valid' => true, 'issues' => $issues, 'recommendations' => $recommendations];
            
        } catch (PdfParserException $e) {
            // FPDI parsing issue - might be compatibility or structural
            // Try to determine if it's just a compatibility issue
            if (strpos($e->getMessage(), 'compression') !== false || 
                strpos($e->getMessage(), 'not supported') !== false) {
                $recommendations[] = 'PDF format compatibility issue but can still be used with fallback methods';
                return ['valid' => true, 'issues' => $issues, 'recommendations' => $recommendations];
            } else {
                // Actual structural issue
                $issues[] = 'PDF parsing error: ' . $e->getMessage();
                $recommendations[] = 'Ensure the file is a valid PDF document';
                return ['valid' => false, 'issues' => $issues, 'recommendations' => $recommendations];
            }
            
        } catch (\Exception $e) {
            // Check if this is a known FPDI compatibility issue
            $errorMessage = $e->getMessage();
            if (strpos($errorMessage, 'compression') !== false || 
                strpos($errorMessage, 'not supported') !== false ||
                strpos($errorMessage, 'FPDI') !== false) {
                // This is likely a compatibility issue, not a structural problem
                $recommendations[] = 'PDF compatibility issue detected but template can still be used with fallback methods';
                return ['valid' => true, 'issues' => $issues, 'recommendations' => $recommendations];
            } else {
                // Unknown error - treat as structural issue
                $issues[] = 'Failed to validate PDF structure: ' . $errorMessage;
                $recommendations[] = 'Ensure the file is a valid PDF document';
                return ['valid' => false, 'issues' => $issues, 'recommendations' => $recommendations];
            }
        }
    }

    /**
     * Perform a quick compatibility test without full processing
     */
    public function quickCompatibilityTest(string $templatePath): array
    {
        $startTime = microtime(true);
        $compatible = $this->checkPdfCompatibility($templatePath);
        $endTime = microtime(true);
        
        return [
            'compatible' => $compatible,
            'testDuration' => round(($endTime - $startTime) * 1000, 2), // milliseconds
            'templateSize' => file_exists($templatePath) ? filesize($templatePath) : 0,
            'templateName' => basename($templatePath)
        ];
    }

    /**
     * Batch validate multiple templates
     */
    public function batchValidate(array $templatePaths): array
    {
        $results = [];
        
        foreach ($templatePaths as $templatePath) {
            $results[basename($templatePath)] = [
                'path' => $templatePath,
                'validation' => $this->validateTemplate($templatePath),
                'quickTest' => $this->quickCompatibilityTest($templatePath)
            ];
        }
        
        return $results;
    }
}